;;; lookup-kanji.el --- Lookup Kanji-Kana Converter
;; Copyright (C) 1999 Lookup Development Team <lookup@ring.gr.jp>

;; Author: TSUCHIYA Masatoshi <tsuchiya@pine.kuee.kyoto-u.ac.jp>
;; Version: $Id: lookup-kanji.el,v 1.1.1.1.4.1 2006/03/18 13:53:03 kazuhiro Exp $

;; This file is part of Lookup.

;; Lookup is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; Lookup is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with Lookup; if not, write to the Free Software Foundation,
;; Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;;; Code:

(require 'evi)
(require 'lookup-utils)
(require 'lookup-vars)

(defcustom lookup-kanji-command "kakasi"
  "*$B4A;z$R$i$,$JJQ49%W%m%0%i%`(B(KAKASI)$B$NL>A0(B"
  :type 'string
  :group 'lookup-kanji)

(defcustom lookup-kanji-option '("-JH" "-p" "-u")
  "*$B4A;z$R$i$,$JJQ49%W%m%0%i%`(B(KAKASI)$B$N%*%W%7%g%s(B"
  :type 'list
  :group 'lookup-kanji)

(defcustom lookup-kanji-coding-system lookup-kakasi-coding-system
  "*$B4A;z$R$i$,$JJQ49%W%m%0%i%`(B(KAKASI)$B$N8F$S=P$7$KMQ$$$kJ8;z%3!<%I(B"
  :type 'symbol
  :group 'lookup-kanji)

(defvar lookup-kanji-process nil
  "$B4A;z$R$i$,$JJQ49%W%m%0%i%`$N%W%m%;%9(B($BFbItJQ?t(B)")

(defvar lookup-kanji-buffer nil
  "$B4A;z$R$i$,$JJQ49%W%m%0%i%`$N%W%m%;%9$r4XO"$E$1$i$l$?%P%C%U%!(B($BFbItJQ?t(B)")

(defun lookup-kanji-generate-buffer ()
  "$B:n6HMQ%P%C%U%!$r@8@.$9$k4X?t(B"
  (unless (and (bufferp lookup-kanji-buffer)
	       (buffer-name lookup-kanji-buffer))
    (setq lookup-kanji-buffer (generate-new-buffer " *lookup-kanji*")
	  lookup-buffer-list (cons lookup-kanji-buffer lookup-buffer-list))
    (buffer-disable-undo lookup-kanji-buffer))
  lookup-kanji-buffer)

(defun lookup-kanji-start-process ()
  "$B4A;z$R$i$,$JJQ49%W%m%0%i%`$r5/F0$9$k4X?t(B"
  (unless (and (processp lookup-kanji-process)
	       (eq (process-status lookup-kanji-process) 'run))
    (lookup-with-coding-system lookup-kanji-coding-system
      (setq lookup-kanji-process
	    (apply 'start-process
		   "lookup-kanji"
		   (lookup-kanji-generate-buffer)
		   lookup-kanji-command
		   lookup-kanji-option)))
    (sit-for 0.1)
    (process-kill-without-query lookup-kanji-process))
  lookup-kanji-process)

(defun lookup-kanji-send-string (str)
  "STR $B$r%W%m%;%9$NI8=`F~NO$KEO$7$F!"$=$N7k2L$rJV$94X?t(B"
  (lookup-process-require (lookup-kanji-start-process) (concat str "\n") "\n")
  (save-excursion
    (set-buffer lookup-kanji-buffer)
    (goto-char (point-min))
    (forward-line 1)
    (prog1 (buffer-substring (point) (progn (end-of-line) (point)))
      (delete-region (point-min) (point-max)))))

(defun lookup-kanji-analysis-output (str)
  "KAKASI $B$N=PNO$K4^$^$l$F$$$k3g8L$r2r@O$7$F%j%9%H$KE83+$9$k(B"
  (if (string-match "{\\([^}]+\\)}" str)
      (let ((x (substring str 0 (match-beginning 0)))
	    (y (substring str (match-end 0))))
	(apply 'nconc 
	       (mapcar 'lookup-kanji-analysis-output
		       (mapcar (lambda (s) (concat x s y))
			       (lookup-split-string (substring str (match-beginning 1) (match-end 1)) "|")))))
    (list str)))

(defun lookup-kanji-get-readings (str)
  "STR $B$r4A;z$R$i$,$JJQ49$7$FF@$i$l$?7k2L$N%j%9%H$rJV$94X?t(B"
  (lookup-kanji-analysis-output (lookup-kanji-send-string str)))

(defun lookup-kanji-filter (str entries)
  "STR $B$,8+=P$7$K4^$^$l$F$$$k$h$&$J%(%s%H%j$N$_$r<h$j=P$9%U%#%k%?4X?t(B"
  (let ((regex (mapconcat 'char-to-string (string-to-char-list string) ".?.?.?")))
    (delq nil (mapcar (lambda (entry)
			(if (string-match regex (lookup-entry-heading entry)) entry))
		      entries))))

(provide 'lookup-kanji)
;;; lookup-kanji.el ends here
