;;; ndest.el --- search agent for Hyper Estraier
;; Copyright (C) 2007 Kazuhiro Ito <kzhr@d1.dion.ne.jp>

;; Author: Kazuhiro Ito <kzhr@d1.dion.ne.jp>

;; ndest.el is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License.

;; ndest.el is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;;; Code:

(require 'lookup)

(defconst ndest-version "0.1")

;;;
;;; Customizable variables
;;;

(defgroup ndest nil
  "Lookup Hyper Estraier interface."
  :group 'lookup-agents)

(defcustom ndest-program-name "estcmd"
  "Program name of estcmd."
  :type 'string
  :group 'ndest)

(defcustom ndest-process-coding-system-for-write 'utf-8
  "Coding system for writing to estcmd process."
  :type 'symbol
  :group 'ndest)

(defcustom ndest-iconv-coding-system-for-write nil
  "An iconv name of coding system for writing to estcmd process. Nil means set automatically."
  :type '(choice (const nil) string)
  :group 'ndest)

(defcustom ndest-max-text 1024
  "*$B8!:w;~$KI=<($9$k%(%s%H%jK\J8$N:GBgD9!#(B
0 $B$r;XDj$9$k$H!"A4J8$rI=<($9$k!#(B
`lookup-max-text'$B$KM%@h$9$k!#(B"
  :type '(choice (const nil) integer)
  :group 'ndest)

(defcustom ndest-follow-link-functions
  '(("^message/rfc822$" nil ndest-follow-link-with-mime-view)
    (nil "/\\.navi2ch/.+\\.dat$" ndest-follow-link-with-navi2ch)
    ("^text/html$" nil ndest-follow-link-with-w3m))
  "A list of arrange functions for content. Each element is a list consist of regexp for data-type, uri and symbol of function."
  :type '(repeat (list (choice :tag "type" regexp (const nil))
		       (choice :tag "URI" regexp (const nil))
		       (choice (function :tag "function")
			       (string :tag "program")
			       (cons :format "%v"
				     :tag "program with args"
				     (string :tag "link-program")
				     (repeat :tag "parameters")
				     ))))
  :group 'ndest)

(defcustom ndest-arrange-functions
  '(("^message/rfc822$" nil ndest-arrange-rfc822)
    ("^text/plain$" "/\\.navi2ch/.+\\.dat$" ndest-arrange-navi2ch-dat))
  "A list of arrange functions for content. Each element is a list consist of regexp for data-type, uri and symbol of function."
  :type '(repeat (list (choice :tag "type" regexp (const nil))
		       (choice :tag "URI" regexp (const nil))
		       (function :tag "arrange-function")))
  :group 'ndest)

(defcustom ndest-program-arguments '("-sf")
  "A list of arguments for estcmd."
  :type '(repeat (string :tag "option"))
  :group 'ndest)

(defcustom ndest-follow-link-from-entry nil
"A string or a vector of symbols and characters meaning a sequence of keystrokes and events for `lookup-entry-follow-ndest-link'. If nil no keystrokes are assigned."
:type '(choice (const nil) (string :tag "Key"))
:group 'ndest)


;;;
;;; types
;;;

(put 'ndest ':methods '(text keyword))

(put 'ndest ':arranges
     '(ndest-arrage-content))

(put 'ndest ':adjusts
     '(lookup-adjust-goto-min))

(put 'ndest ':default-method
     'text)


;;;
;;; Internal variables
;;;

(defvar ndest-mime-raw-buffer nil)

(defvar ndest-mime-view-buffer nil)

(defvar ndest-link-map nil
  "Keymap for ndest links.")

(defconst ndest-coding-system-table
  '(
    ("^euc-jisx0213"      . "EUC-JISX0213")
    ("^japanese-iso-8bit" . "EUC-JP")
    ;; ("^euc-japan-1990"          . "EUC-JP")
    ("^euc-j"             . "EUC-JP")
    ("shift[-_]jisx0213"  . "Shift_JISX0213")
    ("shift[-_]jis"       . "Shift_JIS")
    ("^sjis"              . "Shift_JIS")
    ("^cp932"             . "Shift_JIS")
    ("^utf-8"             . "UTF-8")
    ("^iso-2022-jp"       . "ISO-2022-JP")
    ("^junet"             . "ISO-2022-JP")
    )
  )


;;;
;;; Interface functions
;;;

(put 'ndest 'setup 'ndest-setup)
(defun ndest-setup (agent)
  (let* ((directory (expand-file-name (lookup-agent-location agent))))
    (list (lookup-new-dictionary agent directory "est" "Hyper Estraier"))))

(put 'ndest 'clear 'ndest-clear)
(defun ndest-clear (agent)
  (when (buffer-live-p ndest-mime-raw-buffer)
    (kill-buffer ndest-mime-raw-buffer))
  (when (buffer-live-p ndest-mime-view-buffer)
    (kill-buffer ndest-mime-view-buffer)))

(put 'ndest 'search 'ndest-dictionary-search)
(defun ndest-dictionary-search (dictionary query)
  (with-temp-buffer
    (let ((coding-system-for-write ndest-process-coding-system-for-write)
	  (coding-system-for-read 'utf-8)
	  (args  '("search" "-vs"))
	  ic)
      (if ndest-iconv-coding-system-for-write
	  (setq ic ndest-iconv-coding-system-for-write)
	(setq ic (symbol-name ndest-process-coding-system-for-write))
	(let ((params ndest-coding-system-table)
	      param tmp)
	  (while params
	    (setq param (car params))
	    (when (string-match (car param) ic)
	      (setq ic (cdr param))
	      (setq params nil))
	    (setq params (cdr params)))))
      (setq args 
	    (append
	     args (or (lookup-dictionary-option dictionary ':args t)
		      ndest-program-arguments)
	     (when lookup-max-hits
	       (if (eq lookup-max-hits 0)
		   (list "-max" "-1")
		 (list "-max" (number-to-string lookup-max-hits))))
	     (list "-ic" ic)
	     (let ((max-text
		    (or (lookup-dictionary-option dictionary ':max-text t)
			ndest-max-text
			lookup-max-text)))
	       (when max-text
		 (if (eq max-text 0)
		     nil
		   ;; (list "-sn" "2048" "1024" "1024")
		   (list "-sn" (number-to-string max-text)
			 (number-to-string (/ max-text 2))
			 (number-to-string (/ max-text 2))))))
	     (list (lookup-dictionary-code dictionary)
		   (lookup-query-string query))))
      (apply 'call-process ndest-program-name nil t nil args))
    (goto-char (point-min))
    (when (re-search-forward "^--------\\[[0-9A-F]+\\]--------" nil t)
      (let ((sep (concat "\n" (match-string 0) "\n")))
	(when (search-forward sep nil t)
	  (let (start end end-end entries title type)
	    (setq start (match-end 0))
	    (while (search-forward sep nil t)
	      (setq end (match-beginning 0))
	      (setq end-end (match-end 0))
	      (narrow-to-region start end)
	      (setq title (or (ndest-get-header-string "@title")
			      (ndest-get-filepath-or-uri
			       (ndest-get-header-string "@uri"))))
	      (setq type (ndest-get-header-string "@type"))
	      (setq entries 
		    (cons (lookup-make-entry dictionary (buffer-string)
					     (concat type ":" title))
			  entries))
	      (widen)
	      (goto-char end-end)
	      (setq start end-end))
	    (nreverse entries)))))))
   
(put 'ndest 'content 'ndest-dictionary-content)
(defun ndest-dictionary-content (dictionary entry)
  (lookup-entry-code entry))


;;;
;:: Internal functions
;;;

(defun ndest-initialize ()
  "Initialize ndest module."
  (unless ndest-link-map
    (setq ndest-link-map (copy-keymap lookup-content-mode-map))
    (define-key ndest-link-map "\C-m" 'ndest-follow-link)
    (define-key ndest-link-map "u" 'ndest-show-uri)
    (if (featurep 'xemacs)
	(define-key ndest-link-map 'button2 'ndest-mouse-follow)
      (define-key ndest-link-map [mouse-2] 'ndest-mouse-follow))))

(defun ndest-get-header-string (header)
  (save-excursion
    (save-restriction
      (goto-char (point-min))
      (search-forward "\n\n" nil t)
      (narrow-to-region (point-min) (point))
      (goto-char (point-min))
      (when (re-search-forward
	     (concat "^" (regexp-quote header) "=\\(.+\\)$") nil t)
	(match-string 1)))))

(defun ndest-set-link (start end face type target)
  (let ((binary (list (cons 'type type)
		      (cons 'target target))))
    (add-text-properties start end
			 (list 
			  (if (< emacs-major-version 21) 'local-map
			    'keymap)
			  ndest-link-map
			  'face (or face 'lookup-reference-face)
			  'mouse-face 'highlight
			  'help-echo (format
				      "[%s] mouse-2: play"
				      type)
			  'lookup-tab-stop t
			  'ndest-link binary))))

(defun ndest-get-link (&optional pos)
  (get-text-property (or pos (point)) 'ndest-link))


;;;
;;; Link functions
;;;

(defun lookup-entry-follow-ndest-link ()
  (interactive)
  (let ((dictionary (lookup-entry-dictionary
		     (lookup-entry-current-line-entry))))
    (unless (lookup-entry-content-visible-p)
      (lookup-entry-display-content)))
  (let ((window (get-buffer-window lookup-content-buffer)))
    (and window (select-window window)))
  (switch-to-buffer (lookup-content-buffer))
  (save-excursion
    (goto-char (point-min))
    (let (point)
      (if (get-text-property (point) 'ndest-link)
	  (setq point (point))
	(setq point (next-single-property-change (point) 'ndest-link)))
      (if point
	  (progn
	    (goto-char point)
	    (ndest-follow-link))
	(message "No link")))))

(defun ndest-follow-link ()
  (interactive)
  (let* ((links (ndest-get-link (point)))
	 (type (lookup-assq-ref links 'type))
	 (uri (lookup-assq-ref links 'target))
	 (file (ndest-uri-to-filepath uri))
	 params param fn)
    (setq params
	  (or (lookup-dictionary-option
	       (lookup-entry-dictionary lookup-content-current-entry)
	       ':functions t)
	      ndest-follow-link-functions))
    (catch ':done
      (while params
	(setq param (car params))
	(setq fn (nth 2 param))
	(when (and (or (null (car param))
		       (string-match (car param) type))
		   (or (null (nth 1 param))
		       (string-match (nth 1 param) uri)))
	  (if (symbolp fn)
	      (funcall fn uri file)
	    (when (stringp fn)
	      (setq fn (list fn)))
	    (apply 'start-process " *ndest-links*" nil (car fn) (or file uri) (cdr fn)))
	  (throw  ':done t))
	(setq params (cdr params)))
      (ndest-follow-link-default uri file))))

(defun ndest-show-uri (&optional arg)
  "$B%+!<%=%k0LCV$N%j%s%/@h$N(BURI$B$rI=<($7!"(Bkill-ring$B$KJ]B8$9$k!#(Bprefix argument$B$,$"$j!"(BURI$B$,(Bfile://$B$G;O$^$k>l9g$O%Q%9L>$KJQ49$7$?J8;zNs$,J]B8$5$l$k!#(B"
  (interactive "P")
  (let* ((links (ndest-get-link (point)))
	 (target (lookup-assq-ref links 'target)))
    (when arg
      (setq target (ndest-get-filepath-or-uri target)))
    (message (ndest-unformat target))
    (kill-new target)))

(defun ndest-unformat (string)
  "%$B$NF~$C$?J8;zNs$r$=$N$^$^(Bformat$B7O$N%3%^%s%I$GI=<($G$-$kJ8;zNs$KJQ99$9$k!#(B"
  (with-temp-buffer
    (insert-string string)
    (goto-char (point-min))
    (while (search-forward "%" nil t)
      (insert-string "%"))
    (buffer-string)))

(defun ndest-uri-to-filepath (uri)
  "uri$B$,(Bfile://$B$G;O$^$k>l9g$O%Q%9L>$KJQ49$9$k!#$=$&$G$J$$>l9g$O(Bnil$B$rJV$9!#(B"
  (with-temp-buffer
    (insert-string uri)
    (goto-char (point-min))
    (if (re-search-forward "^file://" nil t)
	(progn
	  (replace-match "")
	  (goto-char (point-min))
	  (when (re-search-forward "^/\\([A-Za-z]\\)|/" nil t)
	    (replace-match "\\1:/" nil nil))
	  (while (re-search-forward "%\\([0-9a-fA-F]\\)\\([0-9a-fA-F]\\)" nil t)
	    (let ((ch (+ (* 16 (string-to-number (match-string 1) 16))
			 (string-to-number (match-string 2) 16))))
	      (delete-region (match-beginning 0) (match-end 0))
	      (goto-char (match-beginning 0))
	      (insert-char ch 1)))
	  (decode-coding-string (buffer-string) 'utf-8))
      nil)))

(defun ndest-get-filepath-or-uri (uri)
  "uri$B$,(Bfile://$B$G;O$^$k>l9g$O%Q%9L>$KJQ49$9$k!#$=$&$G$J$$>l9g$O(Buri$B$rJV$9!#(B"
  (or (ndest-uri-to-filepath uri)
      uri))

(defun ndest-follow-link-with-mime-view (uri file)
  "rfc822$B7A<0$N%U%!%$%k$r(Bmime-view-buffer$B%3%^%s%I$r;H$C$FI=<($9$k!#(BSEMI$B$,I,MW!#(B"
  (require 'mime-view)
  (unless (assq 'lookup-content-mode mime-preview-quitting-method-alist)
    (setq mime-preview-quitting-method-alist
	  (cons '(lookup-content-mode . ndest-mime-view-quit)
		mime-preview-quitting-method-alist)))
  (when file
    (unless (buffer-live-p ndest-mime-raw-buffer)
      (setq ndest-mime-raw-buffer (generate-new-buffer " *ndest-mime-raw*")))
    (unless (buffer-live-p ndest-mime-view-buffer)
      (setq ndest-mime-view-buffer (generate-new-buffer " *ndest-mime-view*")))
    (with-current-buffer
	ndest-mime-raw-buffer
      (widen)
      (delete-region (point-min) (point-max))
      (insert-file-contents file))
    (mime-view-buffer (buffer-name ndest-mime-raw-buffer)
		      (buffer-name ndest-mime-view-buffer))))

(defun ndest-mime-view-quit ()
  (let ((window (get-buffer-window lookup-entry-buffer)))
    (and window (select-window window)))
  (switch-to-buffer lookup-entry-buffer))

(defun ndest-follow-link-with-fiber (uri file)
  "link$B@h$NI=<($K(Bfiber$B$r5/F0$9$k!#(B"
  (start-process nil nil "fiber" (or file uri)))

(defun ndest-follow-link-with-navi2ch (uri file)
  "link$B@h$NI=<($K(Bnavi2ch$B$rMxMQ$9$k!#(B"
  (require 'navi2ch)
  (require 'navi2ch-bookmark)
  (unless navi2ch-bookmark-list
    (navi2ch-bookmark-load-info))
  (if file
      (navi2ch-find-file file)
    (message "Local file is only supported.")))

(defun ndest-follow-link-with-w3m (uri file)
  "link$B@h$NI=<($K(Bw3m$B$rMxMQ$9$k!#(B"
  (require 'w3m)
  (if file 
      (w3m-find-file file)
    (w3m uri)))

(defun ndest-follow-link-default (uri file)
  (if file
      (find-file file)
    (browse-url uri)))
    
;;;
;;; Arrange functions
;;;

(defun ndest-arrage-content (entry)
  (let* ((links (ndest-get-link (point)))
	 (type (ndest-get-header-string "@type"))
	 (uri (ndest-get-header-string "@uri"))
	 params param)
    (setq params ndest-arrange-functions)
    (catch ':done
      (while params
	(setq param (car params))
	(when (and (or (null (car param))
		       (string-match (car param) type))
		   (or (null (nth 1 param))
		       (string-match (nth 1 param) uri)))
	  (funcall (nth 2 param) entry)
	  (throw ':done t))
	(setq params (cdr params)))
      (ndest-arrange-default entry))))

(defun ndest-arrange-rfc822 (entry)
  (let* ((subject (ndest-get-header-string "subject"))
	 (from (ndest-get-header-string "from"))
	 (to (ndest-get-header-string "to"))
	 (uri (ndest-get-header-string "@uri")))
    (search-forward "\n\n" nil t)
    (delete-region (point-min) (match-beginning 0))
    (goto-char (point-min))
    (insert-string (concat subject "\nFrom: " from "\nTo: " to "\n"))
    (let (start end)
      (setq start (point))
      (insert-string uri)
      (setq end (point))
      (ndest-set-link start end nil "message/rfc822" uri))
    (goto-char (point-min))
    (lookup-arrange-default-headings entry)
    (goto-char (point-min))
    (while (re-search-forward "\n\\([^\t\n]+\\)\t[^\n]+\n?" nil t)
      (let ((point (match-beginning 0))
	    (string (match-string 1)))
	(replace-match "" t t)
	(goto-char point)
	(insert-string (concat " " string " "))
	(lookup-make-region-heading point (point) 2)))
    (goto-char (point-min))
    (lookup-arrange-fill-lines entry)))

(defun ndest-arrange-navi2ch-dat (entry)
  (ndest-arrange-plain entry)
  (while (re-search-forward "<\\(br\\|\\)>" nil t)
    (replace-match "\n"))
  (goto-char (point-min))
  (while (re-search-forward "<[^>]*>" nil t)
    (replace-match ""))
  (goto-char (point-min))
  (while (re-search-forward "&\\(gt\\|lt\\|amp\\);" nil t)
    (let ((string (match-string 1)))
      (replace-match 
       (lookup-assoc-ref
	'(("gt"  . ">")
	  ("lt"  . "<")
	  ("amp" . "&"))
	string) t t)))
  (goto-char (point-min))
  (lookup-arrange-fill-lines entry))

(defun ndest-arrange-default (entry)
  (ndest-arrange-plain entry)
  (goto-char (point-min))
  (lookup-arrange-fill-lines entry))

(defun ndest-arrange-plain (entry)
  (let ((uri (ndest-get-header-string "@uri"))
	(type (ndest-get-header-string "@type")))
    (search-forward "\n\n" nil t)
    (delete-region (point-min) (match-beginning 0))
    (goto-char (point-min))
    (let (start end)
      (setq start (point))
      (insert-string uri)
      (setq end (point))
      (ndest-set-link start end nil type uri)))
  (goto-char (point-min))
  (while (re-search-forward "\n\\([^\t\n]+\\)\t[^\n]+\n?" nil t)
    (let ((point (match-beginning 0))
	  (string (match-string 1)))
      (replace-match "" t t)
      (goto-char point)
      (insert-string (concat " " string " "))
      (lookup-make-region-heading point (point) 2))))


;;;
;;; Setup
;;;

(eval-after-load "lookup-content" '(ndest-initialize))

(when ndest-follow-link-from-entry
  (eval-after-load "lookup-entry"
    '(define-key lookup-entry-mode-map ndest-follow-link-from-entry
       (function lookup-entry-follow-ndest-link))))

(provide 'ndest)

;;; ndest.el ends here
