;;; install.el --- Lookup installer -*- coding: utf-8 -*-
;; Copyright (C) 1999 NISHIDA Keisuke <knishida@ring.aist.go.jp>

;; $Id$

;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Code:

(defconst install-elisp-files
  '("lookup.el" "lookup-utils.el" "lookup-types.el"
    "lookup-cache.el" "lookup-vse.el" "lookup-package.el"
    "lookup-kanji.el" "lookup-entry.el" "lookup-content.el"
    "lookup-select.el" "ndic.el" "ndeb.el" "ndtp.el" "ndict.el"
    "ndkks.el" "ndspell.el" "ndcookie.el" "ndmisc.el" "ndsrd.el"
    "ndwnj.el" "sdicf.el" "lookup-vars.el"
    "ndeb-binary.el" "ndebs.el" "ndest.el" "lookup-text.el"
    "lookup-text-table.el" "lookup-kanji-chasen.el"
    "lookup-kanji-kakasi.el" "lookup-kanji-mecab.el"))

(defconst install-info-files
  '("lookup.info" "lookup-guide.info"))

(defconst install-data-files
  '("lookup-logo.xbm" "lookup-logo.png"))

(defvar install-lisp-directory nil)
(defvar install-info-directory nil)
(defvar install-data-directory (expand-file-name "lookup" data-directory))
(defvar install-lookup-version "unknown")

(defun install-check-directory (directory)
  (and (not (file-exists-p directory))
       (y-or-n-p (format "Directory %s is not exist. Create it? " directory))
       (make-directory directory t))
  (if (not (file-directory-p directory))
      (error "%s is not directory" directory))
  (directory-file-name (expand-file-name directory)))

;; get the version number of lookup
(with-current-buffer (get-buffer-create " *work lookup-version*")
  (erase-buffer)
  (setq install-lookup-version
	(condition-case nil
	    (progn
	      (insert-file-contents "VERSION")
	      (goto-char (point-min))
	      (buffer-substring (point-min) (progn (end-of-line) (point))))
	  (error "unknown")))
  (kill-buffer (current-buffer)))

;; message

(if noninteractive nil
  (switch-to-buffer (generate-new-buffer "*Lookup Installer*"))
  (insert "Lookup インストーラ\n")
  (insert "===================\n\n")
  (insert "Lookup のインストールを始めます。")
  (insert "途中で中断するには C-g を押して下さい。\n\n"))

;; directory

(if noninteractive nil
  (insert "ディレクトリの決定\n")
  (insert "------------------\n\n")
  (insert "elisp ファイルのディレクトリを入力して下さい:\n"))
(let ((default "~/emacs/lisp/lookup/"))
  (setq install-lisp-directory
	(install-check-directory
	 (if noninteractive
	     (or (car command-line-args-left) default)
	   (read-file-name "Lisp directory: " default default)))))
(if noninteractive nil
  (insert "    -> " install-lisp-directory "\n\n") (sit-for 0))

(if noninteractive nil
  (insert "info ファイルのディレクトリを入力して下さい:\n"))
(let ((default "~/emacs/info/"))
  (setq install-info-directory
	(install-check-directory
	 (if noninteractive
	     (or (nth 1 command-line-args-left) default)
	   (read-file-name "Info directory: " default default)))))
(if noninteractive nil
  (insert "    -> " install-info-directory "\n\n") (sit-for 0))

(if noninteractive nil
  (insert "画像ファイルのディレクトリを入力して下さい:\n"))
(let ((default install-data-directory))
  (setq install-data-directory
	(install-check-directory
	 (if noninteractive
	     (or (nth 2 command-line-args-left) default)
	   (read-file-name "Image directory: " default default)))))
(if noninteractive nil
  (insert "    -> " install-data-directory "\n\n") (sit-for 0))

;; lookup-vars.el

(if noninteractive nil
  (insert "インストールの実行\n")
  (insert "------------------\n\n")
  (insert "lookup-vars.el.in から lookup-vars.el を生成中...") (sit-for 0))
(with-current-buffer (find-file-noselect "lisp/lookup-vars.el.in" t)
  (if (search-forward "@VERSION@")
      (replace-match install-lookup-version))
  (if (search-forward "@pkgemacsdir@" nil t)	;needs to be fixed to the correct one
      (replace-match install-lisp-directory))
  (write-file "lookup-vars.el")
  (kill-buffer (current-buffer))
  )
(message "Copied lookup-vars.el.in to lookup-vars.el")
(if (not noninteractive) (insert "done\n"))

;; compile

(if noninteractive nil
  (insert "elisp ファイルのコンパイル中...") (sit-for 0))
(add-to-list 'load-path (expand-file-name "lisp"))
(mapc (lambda (file)
	(byte-compile-file
	 (expand-file-name file "lisp")))
      install-elisp-files)
(if (not noninteractive) (insert "done\n"))

;; install

(if noninteractive nil
  (insert "elisp ファイルのインストール中...") (sit-for 0))
(mapcar (lambda (file)
	  (copy-file (expand-file-name file "lisp/")
		     (expand-file-name file install-lisp-directory) t)
	  (message "Installed %s to %s" file install-lisp-directory)
	  (setq file (byte-compile-dest-file file))
	  (copy-file (expand-file-name file "lisp/")
		     (expand-file-name file install-lisp-directory) t)
	  (message "Installed %s to %s" file install-lisp-directory))
	install-elisp-files)
(if (not noninteractive) (insert "done\n"))

(if noninteractive nil
  (insert "info ファイルのフォーマット中...") (sit-for 0))
(mapcar (lambda (info)
	  (if (file-readable-p (expand-file-name info "texi/"))
	      t
	    ;; the pre-formatted .info file does not exist; format it here
	    (let ((texi (concat (file-name-sans-extension info) ".texi"))
		  ;; Workaround for Emacs23 on Windows called from
		  ;; Cygwin shell
		  (shell-file-name (if (or (null (eq system-type 'windows-nt))
					   (file-exists-p shell-file-name))
				       shell-file-name
				     "cmdproxy.exe")))
	      (save-current-buffer
		(let ((buf (find-file-noselect 
			    (expand-file-name texi "texi/") t)))
		  (set-buffer buf)
		  (condition-case nil
		      (texinfo-format-buffer t) ; t for nosplit
		    (error
		     ;; one more try with no @direntry
		     ;; (kill-buffer (current-buffer))
		     (set-buffer buf)
		     (goto-char (point-min))
		     (message "Format failed. Trying no @direntry")
		     (when (re-search-forward "@documentlanguage .*\n" nil t)
		       (replace-match "" t t))
		     (goto-char (point-min))
		     (when (re-search-forward "@direntry" nil t)
		       (setq beg (match-beginning 0))
		       (when (re-search-forward "@end direntry" nil t)
			 (delete-region beg (match-end 0))))
		     (goto-char (point-min))
		     (when (re-search-forward "@dircategory" nil t)
		       (setq end (progn(end-of-line)(point)))
		       (delete-region (match-beginning 0) end))
		     (when (re-search-forward "@detailmenu" nil t)
		       (beginning-of-line)
		       (insert "@c")
		       (when (re-search-forward "@end detailmenu" nil t)
			 (beginning-of-line)
			 (insert "@c")))
		     (goto-char (point-min))
		     (while (re-search-forward "@ifnottex" nil t)
		       (replace-match "@ifinfo" t t))
		     (goto-char (point-min))
		     (while (re-search-forward "@end ifnottex" nil t)
		       (replace-match "@end ifinfo" t t))
		     (goto-char (point-min))
		     (while (re-search-forward 
			     "@\\(email\\|url\\){\\([^}]+\\)}" nil t)
		       (replace-match 
			(buffer-substring (match-beginning 2) (match-end 2))))
		     (texinfo-format-buffer t)))
		  (let ((coding-system-for-write 'iso-2022-jp-unix))
		    (save-buffer 0))
		  (kill-buffer (current-buffer))
		  (kill-buffer buf))))))
	install-info-files)

(if (not noninteractive) (insert "done\n"))

(if noninteractive nil
  (insert "info ファイルのインストール中...") (sit-for 0))
(mapcar (lambda (info)
	  (mapcar (lambda (file)
		    (copy-file (expand-file-name file "texi/")
			       (expand-file-name file install-info-directory)
			       t)
		    (message "Installed %s to %s" file install-info-directory))
		  (directory-files "texi/" nil info)))
	install-info-files)
(if (not noninteractive) (insert "done\n"))

(if noninteractive nil
  (insert "画像 ファイルのインストール中...") (sit-for 0))
(mapc (lambda (file)
	(copy-file (expand-file-name file "lisp/")
		   (expand-file-name file install-data-directory)
		   t)
	(message "Installed %s to %s" file install-data-directory))
      install-data-files)
(if (not noninteractive) (insert "done\n"))

;; clean

(if noninteractive nil
  (insert "一時ファイルの削除中...") (sit-for 0))
(let ((default-directory (expand-file-name "lisp/")))
  (delete-file "lookup-vars.el")
  (mapcar (lambda (file)
	    (delete-file (byte-compile-dest-file (expand-file-name file))))
	  install-elisp-files))
(let ((default-directory (expand-file-name "texi/")))
   (mapcar (lambda (file)
	(delete-file (expand-file-name file)))
	install-info-files))
	
(message "Removed lookup-vars.el, *.elc, *.info")
(if (not noninteractive) (insert "done\n"))

;; initialize

(if noninteractive nil
  (insert "Lookup の初期化中...")
  (sit-for 0)
  (let (lookup-enable-splash)
    (lookup-initialize))
  (insert "done\n")
  (kill-buffer (current-buffer)))

;; congratulate

(if noninteractive
    (message "\nSee etc/SETUP for the setup after installation")
  (switch-to-buffer (generate-new-buffer "*Congratulations!*"))
  (insert "Lookup セットアップガイド
=========================

Lookup のインストールが完了しました。必要に応じて、以下の設定を行なって
下さい。

load-path の設定
----------------

次のようにして load-path の設定を行なって下さい。これはインストールした
プログラムファイルを読み込むために必要なものです。パスが通っていることが
わかっているなら、なくても構いません。

  (setq load-path (cons \"" install-lisp-directory "\" load-path))

info ディレクトリの設定
-----------------------

次のようにして Info-default-directory-list の設定を行なって下さい。これ
はインストールした info ファイルを読み込むために必要なものです。パスが
通っていることがわかっているなら、なくても構いません。

  (setq Info-default-directory-list
        (cons \"" install-info-directory "\" Info-default-directory-list))

dir ファイルの設定
------------------

このインストーラは info ファイルの一覧ファイル(dir ファイル)への書き出し
を自動では行ないません。必要に応じて、次の各行を dir ファイルに追加して
下さい。

* Lookup: (lookup).             Lookup, a Search Interface.
* Lookup Guide: (lookup-guide). The Lookup Startup Guide.

以上でインストールは終了です。ここで C-c C-c とタイプするとスタートアッ
プガイドが開かれますので、まずはそちらからご欄下さい。

-- EOF --
")
  (goto-char (point-min))
  (set-buffer-modified-p nil)
  (view-mode)
  (local-set-key "\C-c\C-c" 'install-open-info))

(defun install-open-info ()
  (interactive)
  (require 'info)
  (Info-find-node (expand-file-name "lookup-guide" install-info-directory)
		  "Top"))

;;; install.el ends here
