/*                                                            -*- C -*-
 * Copyright (c) 2010  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "../config.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <string.h>

#ifdef HAVE_SETMODE
#include <io.h>
#include <fcntl.h>
#endif

#include "defs.h"

/* internal functions */
void output_help ();
void output_version (const char *program_name);
void output_license ();
void output_unsupported (unsigned char ch);

/* comprees.c */
enum dzzip_error_code compress_file
(const char *in_filename, size_t chunk_size);

/* common.c */
int64_t base64_decode_string (const char *string);
void verbose_out (const char *message, ...);

/* decompress.c */
enum dzzip_error_code decompress_file (const char *in_filename, off_t start, size_t length);

/* error.c */
const char *dzzip_error_message (enum dzzip_error_code code);

/* constants */
const char *program_name = "dzzip";

const char *short_options = "dfhklLctvVDs:e:S:E:p:P:C:";
struct option long_options[] = {
  {"decompress", no_argument,       NULL, 'd'},
  {"force",      no_argument,       NULL, 'f'},
  {"help",       no_argument,       NULL, 'h'},
  {"keep",       no_argument,       NULL, 'k'},
  {"list",       no_argument,       NULL, 'l'},
  {"license",    no_argument,       NULL, 'L'},
  {"stdout",     no_argument,       NULL, 'c'},
  {"test",       no_argument,       NULL, 't'},
  {"verbose",    no_argument,       NULL, 'v'},
  {"version",    no_argument,       NULL, 'V'},
  {"debug",      no_argument,       NULL, 'D'},
  {"start",      required_argument, NULL, 's'},
  {"size",       required_argument, NULL, 'e'},
  {"Start",      required_argument, NULL, 'S'},
  {"Size",       required_argument, NULL, 'E'},
  {"pre",        required_argument, NULL, 'p'},
  {"post",       required_argument, NULL, 'P'},
  {"chunk-size", required_argument, NULL, 'C'},
  {NULL, 0, NULL, 0}
};

/* global variables */
int global_verbose = 0;
int global_debug = 0;
int global_keep_original = 0;
int global_overwrite = 0;
int global_stdout = 0;

int main (int argc, char **argv) {
  int i;
  int decompress = 0;
  off_t start = 0;
  size_t length = 0;
  char *tmp_ptr;
  int64_t n;
  enum dzzip_error_code result;
  size_t chunk_size = DICTZIP_DEFAULT_CHUNK_SIZE;

  for (;;) {
    i = getopt_long(argc, argv, short_options, long_options, NULL);
    if (i == -1)
      break;
    switch (i) {
    case 'd':
      decompress = 1;
      break;
    case 'f':
      global_overwrite = 1;
      break;
    case 'h':
      output_help();
      exit(0);
    case 'k':
      global_keep_original = 1;
      break;
    case 'l':
      output_unsupported(i);
      exit(-1);
    case 'L':
      output_unsupported(i);
      exit(0);
    case 'c':
      global_keep_original = 1;
      global_stdout = 1;
      break;
    case 'v':
      global_verbose = 1;
      break;
    case 'D':
      global_debug = 1;
      break;
    case 't':
      output_unsupported(i);
      exit(-1);
    case 'V':
      output_version(program_name);
      exit(0);
    case 'p':
    case 'P':
      output_unsupported(i);
      exit(-1);
    case 'C':
      chunk_size = (off_t)strtoll(optarg, &tmp_ptr, 10);
      if (*tmp_ptr != 0) {
	fprintf(stderr, "Invalid argument %s\n", optarg);
	exit(-1);
      } else if (chunk_size > DICTZIP_MAX_CHUNK_SIZE ||
		 chunk_size < DZZIP_MIN_CHUNK_SIZE) {
	fprintf(stderr, "Illegal chunk size %s. ", optarg);
	fprintf(stderr, "It should be in %d-%d.\n",
		DZZIP_MIN_CHUNK_SIZE, DICTZIP_MAX_CHUNK_SIZE);
	exit(-1);
      }
      break;
    case 's':
      global_keep_original = 1;
      decompress = 1;
      start = (off_t)strtoll(optarg, &tmp_ptr, 10);
      if (*tmp_ptr != 0) {
	fprintf(stderr, "Invalid argument %s\n", optarg);
	exit(-1);
      }
      break;
    case 'e':
      global_keep_original = 1;
      decompress = 1;
      length = (size_t)strtoll(optarg, &tmp_ptr, 10);
      if (*tmp_ptr != 0) {
	fprintf(stderr, "Invalid argument %s\n", optarg);
	exit(-1);
      }
      break;
    case 'S':
      global_keep_original = 1;
      decompress = 1;
      n = base64_decode_string(optarg);
      if (n >= 0) {
	start = n;
      } else {
	exit(-1);
      }
      break;
    case 'E':
      global_keep_original = 1;
      decompress = 1;
      n = base64_decode_string(optarg);
      if (n >= 0) {
	length = n;
      } else {
	exit(-1);
      }
      break;
    default:
      output_help();
      exit(-1);
    }
  }

  if (argc - optind != 1) {
    if (argc == optind)
      fprintf(stderr, "%s: too few arguments\n", program_name);
    else
      fprintf(stderr, "%s: too many arguments\n", program_name);
    output_help();
    exit(-1);
  }

  if (decompress) {
#ifdef HAVE_SETMODE
  if (global_stdout) {
      setmode(fileno(stdout), O_BINARY);
    }
#endif
  result = decompress_file(argv[optind], start, length);
  } else {
    result = compress_file(argv[optind], chunk_size);
  }

  switch (result) {
  case DZZIP_SUCCESS:
    if (!global_keep_original) {
      verbose_out("Deleting input file.");
      if (unlink(argv[optind]) == -1) {
	result = DZZIP_ERR_FILE_IO;
      }
    }
    break;
  case DZZIP_ERR_ZLIB_ERROR:
    break;
  default:
    fputs(dzzip_error_message(result), stderr);
    fputs("\n", stderr);
    break;
  }

  return (int) result;
}

void output_help () {
  printf("Usage: %s [option...] file\n", program_name);
  puts("Options:");
  puts("  -V, --version Show version information.");
  puts("  -h, --help    Show this message.");
  puts("");
  puts("  -d, --decompress        Decompress mode.");
  puts("  -f, --force             Overwrite existing file.");
  puts("  -k, --keep              Do not delete original file.");
  puts("  -c, --stdout            Output to stdout (decompress mode only).");
  puts("  -v, --verbose           Verbose mode.");
  puts("  -D, --debug             Debug mode.");
  puts("  -s, --start=<offset>    Starting offset for decompression (decimal).");
  puts("  -e, --size=<size>       Decompressed size (decimal).");
  puts("  -S, --Start=<offset>    Starting offset for decompression (base64).");
  puts("  -E, --Size=<size>       Decompressed size (base64).");
  printf("  -C  --chunk-size=<size> Chunk size (decimal %d-%d).\n",
	 DZZIP_MIN_CHUNK_SIZE, DICTZIP_MAX_CHUNK_SIZE);
}

void output_version (const char *program_name) {
  printf("%s version %s\n", program_name, VERSION);
  printf("Copyright (C) 2019 Kazuhiro Ito\n");
}

void output_unsupported(unsigned char ch) {
  printf("Option '%c' is unsupported.\n", ch);
}
