/*                                                            -*- C -*-
 * Copyright (c) 2010  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "pdicr_internal.h"

#include <string.h>

/*
  bit status
    0 is set;
   22 is initialized
*/

#define init_flag (0x00400000)
#define set_flag  (0x00000001)
#define all_flags (~(init_flag | set_flag))

void pdicr_entry_initialize (PDICR_Entry *entry) {
  entry->status = init_flag;
  entry->heading = NULL;
  entry->position = 0;
}

void pdicr_entry_finalize (PDICR_Entry *entry) {
  if (pdicr_entry_is_initialized(entry) && entry->heading)
    pdicr_free(entry->heading);
  pdicr_entry_initialize(entry);
}

int pdicr_entry_is_set(PDICR_Entry *entry) {
  return (pdicr_entry_is_initialized(entry) && (entry->status & set_flag))
    ? 1: 0;
}

int pdicr_entry_is_initialized (PDICR_Entry *entry) {
  return  (!(entry->status & all_flags) && (entry->status & init_flag))
    ? 1 : 0;
}

PDICR_Error_Code pdicr_entry_heading
(PDICR_Book *book, PDICR_Entry *entry, char **string) {
  pdicr_log("in: pdicr_entry_heading");

  PDICR_Error_Code ret;

  if (!pdicr_entry_is_set(entry)) {
    ret = PDICR_ERR_INVALID_ENTRY;
    goto finish;
  }

  ret = pdicr_convert_to_utf8(book, entry->heading, strlen(entry->heading),
			      string);
 finish:
  pdicr_log("out: pdicr_entry_heading (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_entry_heading_raw
(PDICR_Book *book, PDICR_Entry *entry, char **string) {
  if (!pdicr_entry_is_set(entry)) return PDICR_ERR_INVALID_ENTRY;

  return pdicr_strdup(entry->heading, string);
}

PDICR_Error_Code pdicr_entry_set
(PDICR_Book *book, PDICR_Entry *entry, const char *heading,
 PDICR_Position position, size_t field_length, size_t field_number_size) {
  pdicr_log("in: pdicr_entry_set (position 0x%x, field length %d)",
	    (int) position, field_length);

  PDICR_Error_Code ret = PDICR_SUCCESS;
  char *string;

  ret = pdicr_convert_from_utf8(book, heading, &string, NULL);
  if (ret == PDICR_SUCCESS)
    ret = pdicr_entry_set_raw(book, entry, string, position, field_length,
			     field_number_size);
  if (string) pdicr_free(string);

  pdicr_log("out: pdicr_entry_set (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_entry_set_raw
(PDICR_Book *book, PDICR_Entry *entry, const char *heading,
 PDICR_Position position, size_t field_length, size_t field_number_size) {
  pdicr_log("in: pdicr_entry_set_raw (position 0x%x, field length %d)",
	    (int) position, field_length);

  PDICR_Error_Code ret = PDICR_SUCCESS;

  pdicr_entry_finalize(entry);

  ret = pdicr_strdup(heading, &entry->heading);
  if (ret == PDICR_SUCCESS) {
    entry->position = position;
    entry->field_length = field_length;
    entry->field_number_size = field_number_size;
    if (entry->heading) entry->status |= set_flag;
  }

  pdicr_log("out: pdicr_entry_set_raw (return %d)", ret);
  return ret;
}
