/*                                                            -*- C -*-
 * Copyright (c) 2010  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "pdicr_internal.h"

#include <string.h>

/*
  bit status
    0 is loaded;
   22 is initialized
*/

#define init_flag (0x00400000)
#define loaded_flag (0x00000001)
#define all_flags (~(init_flag | loaded_flag))

static PDICR_Error_Code pdicr_content_load_internal
(PDICR_Book *book, PDICR_Entry *entry, PDICR_Content *content);

void pdicr_content_initialize (PDICR_Content *content) {
  content->status = init_flag;
  content->heading = NULL;
  content->text = NULL;
  content->pron = NULL;
  content->citation = NULL;
  
  content->level = 0;
  content->important = 0;
  content->modified = 0;
}

void pdicr_content_finalize (PDICR_Content *content) {
  if (pdicr_content_is_loaded(content)) {
    if (content->heading)  pdicr_free(content->heading);
    if (content->text)     pdicr_free(content->text);
    if (content->pron)     pdicr_free(content->pron);
    if (content->citation) pdicr_free(content->citation);
  }
  pdicr_content_initialize(content);
}

int pdicr_content_is_initialized (PDICR_Content *content) {
  return (!(content->status & all_flags) && (content->status & init_flag))
    ? 1: 0;
}

int pdicr_content_is_loaded (PDICR_Content *content) {
  return (pdicr_content_is_initialized(content)
	  && (content->status & loaded_flag)) ? 1: 0;
}

PDICR_Error_Code pdicr_content_load
(PDICR_Book *book, PDICR_Entry *entry, PDICR_Content *content) {
  pdicr_log("in: pdicr_content_load");

  PDICR_Error_Code ret = PDICR_SUCCESS;

  if (!pdicr_book_is_bound(book)) {
    ret = PDICR_ERR_UNBOUND_BOOK;
    goto finish;
  }
  if (!pdicr_entry_is_set(entry)) {
    ret = PDICR_ERR_INVALID_ENTRY;
    goto finish;
  }
  pdicr_content_finalize(content);
  
  ret = pdicr_content_load_internal(book, entry, content);

  if (ret == PDICR_SUCCESS)
    content->status |= loaded_flag;

 finish:
  pdicr_log("out: pdicr_content_load (return %d)", ret);
  return ret;
}

static PDICR_Error_Code pdicr_content_load_internal
(PDICR_Book *book, PDICR_Entry *entry, PDICR_Content *content) {

  pdicr_log("in: pdicr_content_load_internal");

  PDICR_Error_Code ret = PDICR_SUCCESS;
  char *buffer = NULL;
  int bocu_flag = 0, content_attr;
  int heading_length, text_length;
  unsigned char *po;

  ret = pdicr_malloc(entry->field_length + 1, &buffer);
  if (ret != PDICR_SUCCESS) goto finished;
  ret = pdicr_strdup(entry->heading, &content->heading);
  if (ret != PDICR_SUCCESS) goto finished;

  ret = pdicr_book_fread(book, entry->position + 1 + entry->field_number_size,
			entry->field_length, buffer);
  if (ret != PDICR_SUCCESS) goto finished;

  if (book->coding == PDICR_Coding_BOCU1) bocu_flag = 1;
  heading_length = strlen(buffer + bocu_flag);
      
  /* read content attributes */
  content_attr = (unsigned char) buffer[bocu_flag ? 0 : (heading_length + 1)];

  content->level = content_attr & 0x0f;
  content->important = (content_attr & 0x20) ? 1 : 0;
  content->modified = (content_attr & 0x40) ? 1 : 0;

  if (!(content_attr & 0x10)) {
    pdicr_log("Content doesn't have extended structure.");
    /* content does not have extended structure */
    buffer[entry->field_length] = 0;
    if (buffer[heading_length + 2])
      ret = pdicr_strdup(buffer + heading_length + 2, &content->text);
    goto finished;
  }

  pdicr_log("Content has extended structure.");
  /* content has extended structure */
  text_length = strlen(buffer + heading_length + 2);

  if (text_length) {
    ret = pdicr_strdup(buffer + heading_length + 2, &content->text);
    if (ret != PDICR_SUCCESS) goto finished;
  }
  /* read extended part */
  po = (unsigned char *)(buffer + heading_length + 2 + text_length + 1);

  while (*po != 0x80) {

    pdicr_log("Extended attribute is 0x%x", (int)*po);

    if (*po == 0x01 && !content->citation) {
      /* read citation */ 
      ret = pdicr_strdup(po + 1, &content->citation);
      if (ret != PDICR_SUCCESS) goto finished;
    } else if  (*po == 0x02 && !content->pron) {
      /* read pronounciation */ 
      ret = pdicr_strdup(po + 1, &content->pron);
      if (ret != PDICR_SUCCESS) goto finished;
    }

    if (*po >= 0x01 && *po <= 0x04)
      po += strlen(po + 1) + 2;
    else if (*po & 0x10 || *po & 0x40) {
      po += 1 + entry->field_number_size
	+ ((entry->field_number_size == 2) ?
	   pdicr_uint2(po + 1) : pdicr_uint4(po + 1));
    } else {
      ret = PDICR_ERR_UNEXPECTED_DATA;
      goto finished;
    }

    if (content->pron && content->citation) goto finished;
  }

 finished:
  if (buffer) pdicr_free (buffer);

  pdicr_log("out: pdicr_content_load_internal (return %d)", ret);

  return ret;
}

PDICR_Error_Code pdicr_content_heading
(PDICR_Book *book, PDICR_Content *content, char **string) {
  pdicr_log("in: pdicr_content_heading");

  PDICR_Error_Code ret = PDICR_SUCCESS;
  char *tmp_string = NULL;

  if (!pdicr_content_is_loaded(content)) {
    ret = PDICR_ERR_INVALID_CONTENT;
    goto finish;
  }

  if (book->index_order == PDICR_Index_Order_Dictionary) {
    ret = pdicr_heading_heading(content->heading, &tmp_string);
    if (ret != PDICR_SUCCESS) goto finish;
  }
  
  if (tmp_string) {
    ret = pdicr_convert_to_utf8(book, tmp_string, strlen(tmp_string), string);
    pdicr_free(tmp_string); 
  } else {
    ret = pdicr_convert_to_utf8(book, content->heading,
			       strlen(content->heading), string);
  }

  if (ret == PDICR_SUCCESS)
    pdicr_remove_cr(*string);

 finish:
  pdicr_log("out: pdicr_content_heading (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_content_keyword
(PDICR_Book *book, PDICR_Content *content, char **string) {
  pdicr_log("in: pdicr_content_keyword");

  PDICR_Error_Code ret = PDICR_SUCCESS;
  char *tmp_string = NULL;

  if (!pdicr_content_is_loaded(content)) {
    ret = PDICR_ERR_INVALID_CONTENT;
    goto finish;
  }
  if (book->index_order != PDICR_Index_Order_Dictionary) {
    ret = PDICR_ERR_NO_KEYWORD_CONTENT;
    goto finish;
  }

  ret = pdicr_heading_key(content->heading, &tmp_string);
  if (ret != PDICR_SUCCESS) goto finish;
  
  ret = pdicr_convert_to_utf8(book, tmp_string, strlen(tmp_string), string);
  pdicr_free(tmp_string); 

  if (ret == PDICR_SUCCESS)
    pdicr_remove_cr(*string);

 finish:
  pdicr_log("out: pdicr_content_keyword (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_content_heading_raw
(PDICR_Book *book, PDICR_Content *content, char **string) {
  pdicr_log("in: pdicr_content_heading_raw");

  PDICR_Error_Code ret;

  if (!pdicr_content_is_loaded(content)) {
    ret = PDICR_ERR_INVALID_CONTENT;
    goto finish;
  }

  ret = pdicr_strdup(content->heading, string);

 finish:
  pdicr_log("out: pdicr_content_heading_raw (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_content_text
(PDICR_Book *book, PDICR_Content *content, char **string) {
  pdicr_log("in: pdicr_content_text");

  PDICR_Error_Code ret = PDICR_SUCCESS;

  if (!pdicr_content_is_loaded(content)) {
    ret = PDICR_ERR_INVALID_CONTENT;
    goto finish;
  }
  if (!pdicr_content_have_text(content)) {
    ret = PDICR_ERR_NO_TEXT_CONTENT;
    goto finish;
  }

  ret = pdicr_convert_to_utf8(book, content->text,
			      strlen(content->text), string);
  if (ret == PDICR_SUCCESS)
    pdicr_remove_cr(*string);

 finish:
  pdicr_log("out: pdicr_content_text (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_content_text_raw
(PDICR_Book *book, PDICR_Content *content, char **string) {
  pdicr_log("in: pdicr_content_text_raw");

  PDICR_Error_Code ret = PDICR_SUCCESS;

  if (!pdicr_content_is_loaded(content)) {
    ret = PDICR_ERR_INVALID_CONTENT;
    goto finish;
  }
  if (!pdicr_content_have_text(content)) {
    ret = PDICR_ERR_NO_TEXT_CONTENT;
    goto finish;
  }

  ret = pdicr_strdup(content->text, string);

 finish:
  pdicr_log("out: pdicr_content_text_raw (return %d)" ,ret);
  return ret;
}

PDICR_Error_Code pdicr_content_pron (PDICR_Book *book, PDICR_Content *content,
				   char **string) {
  pdicr_log("in: pdicr_content_pron");

  PDICR_Error_Code ret = PDICR_SUCCESS;

  if (!pdicr_content_is_loaded(content)) {
    ret = PDICR_ERR_INVALID_CONTENT;
    goto finish;
  }
  if (!pdicr_content_have_pron(content)) {
    ret = PDICR_ERR_NO_PRON_CONTENT;
    goto finish;
  }

  ret = pdicr_convert_from_pron(book, content->pron, string);

  if (ret == PDICR_SUCCESS)
    pdicr_remove_cr(*string);

 finish:
  pdicr_log("out: pdicr_content_pron (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_content_pron_raw
(PDICR_Book *book, PDICR_Content *content, char **string) {
  pdicr_log("in: pdicr_content_pron_raw");

  PDICR_Error_Code ret;

  if (!pdicr_content_is_loaded(content)) {
    ret = PDICR_ERR_INVALID_CONTENT;
    goto finish;
  }
  if (!pdicr_content_have_pron(content)) {
    ret = PDICR_ERR_NO_PRON_CONTENT;
    goto finish;
  }

  ret = pdicr_strdup(content->pron, string);

 finish:
  pdicr_log("out: pdicr_content_pron_raw (return %d)");
  return  ret;
}

PDICR_Error_Code pdicr_content_citation
(PDICR_Book *book, PDICR_Content *content, char **string) {
  pdicr_log("in: pdicr_content_citation");

  PDICR_Error_Code ret = PDICR_SUCCESS;

  if (!pdicr_content_is_loaded(content)) {
    ret = PDICR_ERR_INVALID_CONTENT;
    goto finish;
  }
  if (!pdicr_content_have_citation(content)) {
    ret = PDICR_ERR_NO_CITATION_CONTENT;
    goto finish;
  }

  ret = pdicr_convert_to_utf8(book, content->citation,
			      strlen(content->citation), string);

  if (ret == PDICR_SUCCESS)
    pdicr_remove_cr(*string);

 finish:
  pdicr_log("out: pdicr_content_citation (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_content_citation_raw
(PDICR_Book *book, PDICR_Content *content, char **string) {
  pdicr_log("in: pdicr_content_citation_raw");

  PDICR_Error_Code ret;

  if (!pdicr_content_is_loaded(content)) {
    ret = PDICR_ERR_INVALID_CONTENT;
    goto finish;
  }
  if (!pdicr_content_have_citation(content)) {
    ret = PDICR_ERR_NO_CITATION_CONTENT;
    goto finish;
  }

  ret = pdicr_strdup(content->citation, string);

 finish:
  pdicr_log("out: pdicr_content_citation_raw (return %d)", ret);
  return ret;
}

int pdicr_content_have_text (PDICR_Content *content) {
  return content->text ? 1 : 0;
}

int pdicr_content_have_citation (PDICR_Content *content) {
  return content->citation ? 1 : 0;
}

int pdicr_content_have_pron (PDICR_Content *content) {
  return content->pron ? 1 : 0;
}

int pdicr_content_is_important (PDICR_Content *content) {
  return content->important ? 1 : 0;
}

int pdicr_content_is_modified (PDICR_Content *content) {
  return content->modified ? 1 : 0;
}

int pdicr_content_level (PDICR_Content *content) {
  return content->level;
}
