/*                                                            -*- C -*-
 * Copyright (c) 2010  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "pdicr_internal.h"

#include <string.h>
#include <stdarg.h>

PDICR_Error_Code pdicr_malloc (size_t size, char **buffer) {
  pdicr_log("in: pdicr_malloc");

  PDICR_Error_Code ret = PDICR_SUCCESS;

  *buffer = malloc(size);
  if (!*buffer) {
    ret = PDICR_ERR_MEMORY_EXHAUSTED;
  }
  pdicr_log("out: pdicr_malloc (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_realloc (size_t size, char **buffer) {
  pdicr_log("in: pdicr_realloc");

  PDICR_Error_Code ret = PDICR_SUCCESS;
  void *new_buffer;

  new_buffer = (void *)realloc(*buffer, size);
  if (!new_buffer) {
    ret = PDICR_ERR_MEMORY_EXHAUSTED;
  } else {
    *buffer = new_buffer;
  }

  pdicr_log("out: pdicr_realloc (return %d)", ret);
  return ret;
}

void pdicr_free(char *buffer) {
  pdicr_log("in: pdicr_free");

  free(buffer);

  pdicr_log("out: pdicr_free");
}

/* Remove CR from NULL terminated text. */
/* Return value is a number of removed CR. */
int pdicr_remove_cr (char *string) {
  pdicr_log("in: pdicr_remove_cr");

  int dst, src, ret;

  dst = 0;
  src = 0;

  while (1) {
    while (string[src] == 0x0d)
      src++;
    if (src != dst)
      string[dst] = string[src];
    if (!string[src])
      break;
    src++, dst++;
  }

  ret = src - dst;

  pdicr_log("out: pdicr_remove_cr (result %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_strdup (const char *string, char **to_s) {
  pdicr_log("in: pdicr_strdup");

  PDICR_Error_Code ret = PDICR_SUCCESS;
  char *tmp;

  if (!string || !strlen(string)) {
    *to_s = NULL;
    goto finish;
  }

  tmp = strdup(string);
  if (!tmp) {
    ret = PDICR_ERR_MEMORY_EXHAUSTED;
    goto finish;
  }
  *to_s = tmp;

 finish:
  pdicr_log("out: pdicr_strdup (result %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_heading_key (const char *heading, char **result) {
  pdicr_log("in: pdicr_heading_key");

  PDICR_Error_Code ret = PDICR_SUCCESS;
  int count = 0;

  for (; count < strlen(heading); count++)
    if (heading[count] == '\t') break;

  ret = pdicr_malloc(count + 1, result);
  if (ret != PDICR_SUCCESS) goto finish;
  memcpy(*result, heading, count);
  (*result)[count] = 0;

 finish:
  pdicr_log("out: pdicr_heading_key (result %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_heading_heading (const char *heading, char ** result) {
  pdicr_log("in: pdicr_heading_heading");

  int count, tmp = strlen(heading);
  char *buffer;

  for (count = 0; count < tmp; count++)
    if (heading[count] == '\t') break;
  if (count == tmp)
    count = -1;
  PDICR_Error_Code ret = pdicr_strdup(heading + count + 1, result);

 finish:
  pdicr_log("out: pdicr_heading_heading (result %d)", ret);
  return ret;
}


/*return base name of filename from path name */
PDICR_Error_Code pdicr_path_basename (const char *path, char **basename) {
  pdicr_log("in: pdicr_path_basename");

  PDICR_Error_Code ret = PDICR_SUCCESS;
  int start, end, last;

  for (last = strlen(path) - 1; last > 0; last--) {
    if (path[last] != '/') break;
  }

  for (start = last; start >= 0; start--) {
    if (path[start] == '/') break;
  }
  start++;

  for (end = last; end >= start; end--) {
    if (path[end] == '.') break;
  }

  if (start > end) end = last;

  ret = pdicr_malloc(end - start + 1, basename);
  if (ret != PDICR_SUCCESS) goto finish;

  memcpy(*basename, path + start, end - start);
  (*basename)[end - start] = 0;

 finish:
  pdicr_log("out: pdicr_path_basename (result %d)", ret);
  return ret;
}


/* 0; un-initialized
   1; enabled
   2: disabled
*/

static int pdicr_log_flag = 0;

void pdicr_log_initialize (void) {
  pdicr_log_flag = (getenv("PDICR_DEBUG") != NULL) ? 1 : 2;
}

void pdicr_log (const char *message, ...) {
  va_list ap;
  va_start(ap, message);

  if (!pdicr_log_flag) {
    pdicr_log_initialize();
  }

  if (pdicr_log_flag == 1) {
    fputs("[PDICR] ", stderr);

    vfprintf(stderr, message, ap);
    fputc('\n', stderr);
    fflush(stderr);
  }
}
