;;; lookup-cache.el --- lookup cache management.
;; Copyright (C) 2010  Kazuhiro Ito <kzhr@d1.dion.ne.jp>

;; Author: Kazuhiro Ito <kzhr@d1.dion.ne.jp>
;; Version: $Id: lookup-cache.el,v 1.1.2.5 2010-06-30 04:05:04 kazuhiro Exp $

;; This file is part of Lookup.

;; Lookup is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; Lookup is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with Lookup; if not, write to the Free Software Foundation,
;; Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;;; Code:

;; Customizable variables.
(defcustom lookup-cache-file-name
  (concat "~" init-file-user "/.lookup-cahce")
  "File name for Lookup cache.  When nil file is not used."
  :type '(choice file
		 (const nil))
  :group 'lookup-setup-variables)

(defcustom lookup-cache-file-coding-system
  (or (and (coding-system-p 'utf-8-emacs-unix)
	   'utf-8-emacs-unix)
      'emacs-mule)
  "Coding system for Lookup cache file."
  :type 'coding-system
  :group 'lookup-setup-variables)

(defcustom lookup-enable-cache
  t
  "Whether enable Lookup cache system.  Storing and saving cache data are always going on even if value is nil."
  :type 'boolean
  :group 'lookup-setup-variables)

;; Internal variables.
(defvar lookup-cache-cache nil
  "Lookup cache data")

;; Interface functinos.
(defun lookup-cache-get (class location dictionary key)
  "Check cache data of specified storage and key, then return cached value.  Return symbol not-cached when not cached.  If `lookup-enable-cache' is nil, always return symbol not-cached.
CLASS, LOCATION, DICTIONARY are combination of (symbol, string, string), (symbol, string, nil), (symbol, nil, nil), or (nil, nil, nil).  KEY is symbol."
  (let* ((class-cache (cdr (assoc (symbol-name class) lookup-cache-cache)))
	 (location-cache (cdr (assoc location class-cache)))
	 (dictionary-cache (cdr (assoc dictionary location-cache)))
	 result)
    (cond
     ((null lookup-enable-cache)
      (setq result nil))
     (dictionary
      (setq result (assq key dictionary-cache)))
     (location
      (setq result (assq key location-cache)))
     (class
      (setq result (assq key class-cache)))
     (t
      (setq result (assq key lookup-cache-cache))))
    (if result (cdr result)
      'not-cached)))

(defun lookup-cache-put (class location dictionary key value)
  "Store cache data in specified storage and key.  Do not store symbol not-cached because `lookup-cache-get' returns that value when cache is not hit.
CLASS, LOCATION, DICTIONARY are combination of (symbol, string, string), (symbol, string, nil), (symbol, nil, nil), or (nil, nil, nil).  KEY is symbol."
  (let* ((class (and class (symbol-name class)))
	 (class-cache (assoc class lookup-cache-cache))
	 (location-cache (assoc location (cdr class-cache)))
	 (dictionary-cache (assoc dictionary (cdr location-cache))))
    (cond
     (dictionary
      (setq dictionary-cache
	    (cons dictionary (cons (cons key value)
				   (delq (assq key dictionary-cache)
					 (cdr dictionary-cache))))))
     (location (setq dictionary key
		     dictionary-cache (cons key value)))
     (class (setq location key
		  location-cache (cons key value)))
     (t (setq class key
	      class-cache (cons key value))))
    (when dictionary-cache
      (setq location-cache
	    (cons location (cons dictionary-cache
				 (delq (assoc dictionary location-cache)
				       (cdr location-cache))))))
    (when location-cache
      (setq class-cache
	    (cons class (cons location-cache
			      (delq (assoc location class-cache)
				    (cdr class-cache))))))
    (setq lookup-cache-cache
	  (cons class-cache
		(delq (assoc class lookup-cache-cache)
		      lookup-cache-cache))))
  value)

(defun lookup-cache-clear (&optional class location dictionary key)
  "Clear Lookup cache data.  You can clear only specific storage by optional arguments."
  (interactive)
  (let* ((class (and class (symbol-name class)))
	 (class-cache (assoc class lookup-cache-cache))
	 (location-cache (assoc location (cdr class-cache)))
	 (dictionary-cache (assoc dictionary (cdr location-cache))))
    (cond
     (dictionary
      (and (setq dictionary-cache (and key (delq (assq key dictionary-cache)
						 (cdr dictionary-cache))))
	   (setq dictionary-cache (cons dictionary dictionary-cache))))
     (location
      (and (setq location-cache (and key (delq (assq key location-cache)
					       (cdr location-cache))))
	   (setq location-cache (cons location location-cache))))
     (class
      (and (setq class-cache (delq (assq key class-cache) (cdr class-cache)))
	   (setq class-cache (cons class class-cache))))
     (t
      (setq lookup-cache-cache (and key (delq (assq key lookup-cache-cache)
					      lookup-cache-cache)))))
    (when dictionary
      (and (setq location-cache
		 (delq nil (cons dictionary-cache
				 (delq (assoc dictionary location-cache)
				       (cdr location-cache)))))
	   (setq location-cache (cons location location-cache))))
    (when location
      (and (setq class-cache
		 (delq nil (cons location-cache
				 (delq (assoc location class-cache)
				       (cdr class-cache)))))
	   (setq class-cache (cons class class-cache))))
    (when class
      (setq lookup-cache-cache
	    (delq nil (cons class-cache
			    (delq (assoc class lookup-cache-cache)
				  (cdr lookup-cache-cache))))))))

(defun lookup-cache-save ()
  "Save Lookup cache data to file specified by `lookup-cache-file-name' unless it is nil."
  (interactive)
  (when lookup-cache-file-name
    (let ((coding-system-for-write lookup-cache-file-coding-system))
      (with-temp-file (expand-file-name lookup-cache-file-name)
	(insert (format ";; -*- coding: %s -*-\n"
			lookup-cache-file-coding-system))
	(insert ";; This file is generated automatically by Lookup.\n\n")
	(pp lookup-cache-cache (current-buffer))))))

(defun lookup-cache-load ()
  "Load Lookup cache data from file specified by `lookup-cache-file-name' unless it is nil."
  (when lookup-cache-file-name
    (let ((file-name (expand-file-name lookup-cache-file-name)))
      (when (file-exists-p file-name)
	(setq lookup-cache-cache
	      (read (with-temp-buffer
		      (insert-file-contents file-name)
		      (buffer-string))))))))

(defmacro lookup-cache-try (class location dictionary key &rest body)
  "Try `lookup-cache-get' and return cached value if cached.  When not cached or `lookup-enable-cache' is nil, evaluate BODY and cache, return result."
  `(if (eq (lookup-cache-get ,class ,location ,dictionary ,key)
	   'not-cached)
       (lookup-cache-put ,class ,location ,dictionary ,key
			 (progn ,@body))
     (lookup-cache-get ,class ,location ,dictionary ,key)))

;;;

(provide 'lookup-cache)

;;; lookup-cache.el ends here
