/*                                                            -*- C -*-
 * Copyright (c) 2010  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "pdicr_internal.h"

#include <string.h>

/*
  bit status
  000 is loaded;
  022 is initialized
*/

#define init_flag (0x00400000)
#define loaded_flag (0x00000001)
#define all_flags (~(init_flag | loaded_flag))

static void pdicr_cache_purge (PDICR_Book *book);
static void pdicr_cache_initialize (PDICR_Cache *cache);
static PDICR_Error_Code pdicr_caches_cache
(PDICR_Book *book, PDICR_Position pos, PDICR_Cache **result);
static PDICR_Error_Code pdicr_cache_create
(PDICR_Book *book, PDICR_Position pos);


static void pdicr_cache_initialize (PDICR_Cache *cache) {
  cache->status = init_flag;

  cache->next = NULL;
  cache->prev = NULL;
  cache->position = 0;
  cache->size = 0;
}

PDICR_Error_Code pdicr_caches_fread
(PDICR_Book *book, PDICR_Position pos, size_t req_size, char *buffer,
 size_t *result) {
  PDICR_Error_Code ret = PDICR_SUCCESS;
  PDICR_Cache *cache;
  size_t read, cp;

  if (req_size > (PDICR_CACHE_SIZE * 2)) {
  /* if (req_size) { */
    if (fseek(book->file, pos, SEEK_SET)) return PDICR_ERR_FSEEK;
    read = fread(buffer, 1, req_size, book->file);
    if (result) *result = read;
    return PDICR_SUCCESS;
  }

  read = 0;
  while (req_size) {
    ret = pdicr_caches_cache(book, pos, &cache);
    if (ret != PDICR_SUCCESS) return ret;
    cp = cache->size - (pos - cache->position);
    if (cp > req_size) cp = req_size;
    memcpy(buffer, cache->buffer + (pos - cache->position), cp);
    req_size -= cp;
    pos += cp;
    buffer += cp;
    read += cp;
    if (cache->size != PDICR_CACHE_SIZE) break;
  }

  if (result) *result = read;
  return ret;
}


static PDICR_Error_Code pdicr_caches_cache
(PDICR_Book *book, PDICR_Position pos, PDICR_Cache **result) {
  PDICR_Error_Code ret;
  PDICR_Cache *cache;
  PDICR_Position target;

  cache = book->cache;
  target = pos - (pos % PDICR_CACHE_SIZE);

  while (cache) {
    if (cache->position == target) {
      if (cache->prev) {
	cache->prev->next = cache->next;
	if (cache->next)
	  cache->next->prev = cache->prev;
	cache->next = book->cache;
	cache->prev = NULL;
	book->cache = cache;
      }
      *result = cache;
      return PDICR_SUCCESS;
    }
     cache = cache->next;
  }
  ret = pdicr_cache_create (book, target);
  if (ret != PDICR_SUCCESS) return ret;
  pdicr_cache_purge(book);
  *result = book->cache;

  return ret;
}

static PDICR_Error_Code pdicr_cache_create
(PDICR_Book *book, PDICR_Position pos) {
  PDICR_Error_Code ret = PDICR_SUCCESS;
  PDICR_Cache *cache;

  ret = pdicr_malloc(sizeof(PDICR_Cache), (char **)&cache);
  if (ret != PDICR_SUCCESS) return ret;

  pdicr_cache_initialize(cache);

  if (fseek(book->file, pos, SEEK_SET)) {
    ret = PDICR_ERR_FSEEK;
    goto failed;
  }

  cache->size = fread(cache->buffer, 1, PDICR_CACHE_SIZE, book->file);
  cache->status |= loaded_flag;
  cache->position = pos;

  cache->next = book->cache;
  if (book->cache) book->cache->prev = cache;
  book->cache = cache;

  return ret;

 failed:
  if (cache) pdicr_free((char *)cache);
  return ret;
}

static void pdicr_cache_purge (PDICR_Book *book) {
  int count;
  PDICR_Cache *cache, *next;

  count = 0, cache = book->cache;
  while ((count + 1 < PDICR_CACHE_NUMBER) && cache) {
    count++;
    cache = cache->next;
  }

  if (cache && cache->next) {
    pdicr_caches_finalize (cache->next);
    cache->next = NULL;
  }
}

void pdicr_caches_finalize (PDICR_Cache *caches) {
  PDICR_Cache *next;

  while (caches) {
    next = caches->next;
    pdicr_free((char *)caches);
    caches = next;
  }
}
