/*                                           -*- mode: C; coding: utf-8; -*-
 * Copyright (c) 2010  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "pdicr_internal.h"
#include "bocu1.h"

#include <string.h>

static PDICR_Pron_Table pdicr_pron_table_silipa93 = {
  /* 0x00 - 0x1f */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
  /* 0x20 - 0x3f */
  " ", "̋", "ı", "̄", "̀", "̏", "̌", "ʼ", 
  "̆", "̃", "̊", "̟", ",", "̤", ".", "/", 
  "̰", "̘", "̙", "̝", "̞", "̪", "̻", "̜", 
  "̥", "̯", "ˡ", "ʟ", "ⁿ", "̠", "ˑ", "ʔ", 
  /* 0x40 - 0x5f */
  "́", "ɑ", "β", "ç", "ð", "ɛ", "ɤ", "ɢ", 
  "ʰ", "ɪ", "ʲ", "ʜ", "ɮ", "ɱ", "ŋ", "ø", 
  "ɵ", "æ", "ɾ", "ʃ", "θ", "ʊ", "ʋ", "ʷ", 
  "χ", "ʏ", "ʒ", "[", "\\", "]", "̂", "̈", 
  /* 0x60 - 0x7f */
  "̩", "a", "b", "c", "d", "e", "f", "ɡ", 
  "h", "i", "j", "k", "l", "m", "n", "o", 
  "p", "q", "r", "s", "t", "u", "v", "w", 
  "x", "y", "z", "ʀ", "̚", "ɽ", "̽", "", 
  /* 0x80 - 0x9f */
  "˩˧", "ɒ", "ɘ", "͡", "∥", "˥˧", "˥˩", "ʘ", 
  NULL, NULL, "˥", "ꜛ", "ɐ", "ɔ", "ǀ", NULL, 
  NULL, "˦", "ǁ", NULL, NULL, "˧", "|", "ǃ", 
  NULL, NULL, "˨", "ꜜ", "ǂ", NULL, NULL, "˩", 
  /* 0xa0 - 0xbf */
  NULL, NULL, NULL, NULL, "̬", NULL, "̹", "ʂ", 
  "ɹ", "ɠ", NULL, "ə", "ʉ", NULL, NULL, "ɶ", 
  "̺", NULL, "ɴ", "ˤ", "ʎ", "ɯ", NULL, NULL, 
  "ɸ", "ʢ", "ɓ", NULL, NULL, "ʐ", NULL, "œ", 
  /* 0xc0 - 0xdf */
  "ʕ", NULL, "ɬ", "ʌ", "ɣ", NULL, "ʝ", "ˌ", 
  "ˈ", "ᵐ", " ", "ᶮ", "↗", "↘", "ɜ", "ɞ", 
  NULL, "̼", "ʁ", "ɻ", "ᵑ", "˞", "-", "ʄ", 
  "˧˥", "˧˩", NULL, NULL, NULL, NULL, NULL, NULL, 
  /* 0xe0 - 0xff */
  NULL, NULL, NULL, "ʍ", "ɺ", "ɰ", NULL, "ɥ", 
  "˩˥", NULL, "ɖ", "ɗ", "ˠ", "‿", "ɧ", "ɟ", 
  "ħ", "ɭ", "̴", NULL, NULL, "ʙ", "ɨ", "ɳ", 
  "ɲ", "ː", "ɦ", "ʡ", "ʑ", "ʛ", "ɕ", "ʈ"
};

PDICR_Error_Code pdicr_convert_from_pron
(PDICR_Book *book, const char *string, char **to_s) {
  PDICR_Error_Code ret = PDICR_SUCCESS;
  char *buffer;
  int count, consumed, buffer_size, str_length;
  PDICR_Pron_Table *table;

  switch (book->pron_table_code) {
  case PDICR_Pron_Table_Code_ASIS:
    return pdicr_convert_to_utf8(book, string, strlen(string), to_s);
  case PDICR_Pron_Table_Code_SILIPA93:
    table = &pdicr_pron_table_silipa93;
    break;
  default:
    return PDICR_ERR_INVALID_PRON_TABLE_CODE;
  }

  if (book->coding == PDICR_Coding_BOCU1)
    return pdicr_convert_bocu1_pron(string, table, to_s);

  buffer_size = strlen(string) * 2;
  ret = pdicr_malloc(buffer_size, &buffer);
  if (ret != PDICR_SUCCESS) return ret;

  for (count = 0, consumed = 0; string[count]; count++) {
    if (table->code[(unsigned char)string[count]]) {
      str_length = strlen(table->code[(unsigned char)string[count]]);
      while ((buffer_size - consumed) < str_length) {
	buffer_size += buffer_size;
	ret = pdicr_realloc(buffer_size, &buffer);
	if (ret != PDICR_SUCCESS) goto failed;
      }
      memcpy(buffer + consumed, table->code[(unsigned char)string[count]],
	     str_length);
      consumed += str_length;
    } else {
      while ((buffer_size - consumed) <
	     utf8_length((unsigned char)string[count])) {
	buffer_size += buffer_size;
	ret = pdicr_realloc(buffer_size, &buffer);
	if (ret != PDICR_SUCCESS) goto failed;
      }
      consumed += pdicr_write_utf8((unsigned char)string[count],
				  buffer + consumed);
    }
  }
  ret = pdicr_realloc(consumed + 1, &buffer);
  if (ret != PDICR_SUCCESS) goto failed;
  buffer[consumed] = 0;
  *to_s = buffer;
  return ret;

 failed:
  if (buffer) pdicr_free(buffer);
  return ret;
}


/* Convert BOCU-1 string to utf-8 via indicated conversion table. */
PDICR_Error_Code pdicr_convert_bocu1_pron
(const char *string, PDICR_Pron_Table *table, char **to_s) {
  PDICR_Error_Code ret = PDICR_SUCCESS;
  Bocu1Rx Rx;
  char *buffer;
  int res, consumed = 0;
  size_t buffer_length;

  Rx.prev = 0;
  Rx.count = 0;
  Rx.diff = 0;

  buffer_length = strlen(string) * 2;
  ret = pdicr_malloc(buffer_length, &buffer);
  if (ret != PDICR_SUCCESS) return ret;

  while (*string != 0) {
    res = decodeBocu1(&Rx, (unsigned char)*string);
    if (res < -1) {
      ret = PDICR_ERR_BOCU1_DECODE_FAILURE;
      goto finished;
    }
    string++;
    if (res == -1)
      continue;

    if (res <= 0xff && table->code[res]) {
      while ((buffer_length - consumed) < strlen(table->code[res])) {
	buffer_length += buffer_length;
	ret = pdicr_realloc(buffer_length, &buffer);
	if (ret != PDICR_SUCCESS) goto finished;;
      }
      memcpy(buffer + consumed, table->code[res], strlen(table->code[res]));
      consumed += strlen(table->code[res]);
    } else {
      while ((buffer_length - consumed) < utf8_length(res)) {
	buffer_length += buffer_length;
	ret = pdicr_realloc(buffer_length, &buffer);
	if (ret != PDICR_SUCCESS) goto finished;;
      }
      consumed += pdicr_write_utf8(res, buffer + consumed);
    }
  }

  ret = pdicr_malloc(consumed + 1, to_s);
  if (ret != PDICR_SUCCESS) goto finished;

  memcpy(*to_s, buffer, consumed);
  (*to_s)[consumed] = 0;

 finished:
  if (buffer) pdicr_free(buffer);
  return ret;
}
