/*                                                            -*- C -*-
 * Copyright (c) 2010  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "pdicr_internal.h"

#include <string.h>

PDICR_Error_Code pdicr_malloc (size_t size, char **buffer) {
  *buffer = malloc(size);
  if (!*buffer) return PDICR_ERR_MEMORY_EXHAUSTED;
  return PDICR_SUCCESS;
}

PDICR_Error_Code pdicr_realloc (size_t size, char **buffer) {
  void *new_buffer;

  new_buffer = (void *)realloc(*buffer, size);
  if (!new_buffer) return PDICR_ERR_MEMORY_EXHAUSTED;
  *buffer = new_buffer;
  return PDICR_SUCCESS;
}

void pdicr_free(char *buffer) {
  free(buffer);
}

/* Remove CR from NULL terminated text. */
/* Return value is a number of removed CR. */
int pdicr_remove_cr (char *string) {
  int dst,src;

  dst = 0;
  src = 0;

  while (1) {
    while (string[src] == 0x0d)
      src++;
    if (src != dst)
      string[dst] = string[src];
    if (!string[src])
      break;
    src++, dst++;
  }
  return (src - dst);
}

PDICR_Error_Code pdicr_strdup (const char *string, char **to_s) {
  char *tmp;

  if (!string || !strlen(string)) {
    *to_s = NULL;
    return PDICR_SUCCESS;
  }

  tmp = strdup(string);
  if (!tmp) return PDICR_ERR_MEMORY_EXHAUSTED;
  *to_s = tmp;
  return PDICR_SUCCESS;
}

PDICR_Error_Code pdicr_heading_key (const char *heading, char **result) {
  PDICR_Error_Code ret = PDICR_SUCCESS;
  int count = 0;

  for (; count < strlen(heading); count++)
    if (heading[count] == '\t') break;

  ret = pdicr_malloc(count + 1, result);
  if (ret != PDICR_SUCCESS) return ret;
  memcpy(*result, heading, count);
  (*result)[count] = 0;

  return ret;
}

PDICR_Error_Code pdicr_heading_heading (const char *heading, char ** result) {
  PDICR_Error_Code ret = PDICR_SUCCESS;
  int count, tmp = strlen(heading);
  char *buffer;

  for (count = 0; count < tmp; count++)
    if (heading[count] == '\t') break;
  if (count == tmp) count = -1;
  
  ret = pdicr_strdup(heading + count + 1, result);

  return ret;
}


/*return base name of filename from path name */
PDICR_Error_Code pdicr_path_basename (const char *path, char **basename) {
  PDICR_Error_Code ret = PDICR_SUCCESS;
  int start, end, last;

  for (last = strlen(path) - 1; last > 0; last--) {
    if (path[last] != '/') break;
  }

  for (start = last; start >= 0; start--) {
    if (path[start] == '/') break;
  }
  start++;

  for (end = last; end >= start; end--) {
    if (path[end] == '.') break;
  }

  if (start > end) end = last;

  ret = pdicr_malloc(end - start + 1, basename);
  if (ret != PDICR_SUCCESS) return ret;

  memcpy(*basename, path + start, end - start);
  (*basename)[end - start] = 0;

  return ret;
}
