/*                                                            -*- C -*-
 * Copyright (c) 2010  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "pdicr_internal.h"

#include <string.h>

/*
  bit status
    0 is bound;
   22 is initialized
*/

#define init_flag (0x00400000)
#define bound_flag (0x00000001)
#define all_flags (~(init_flag | bound_flag))

/* static functions */
static PDICR_Error_Code bind_internal (PDICR_Book *book);

int pdicr_book_is_bound (PDICR_Book *book) {
  return (pdicr_book_is_initialized(book) && (book->status & bound_flag))
    ? 1 : 0;
}

int pdicr_book_is_initialized (PDICR_Book *book) {
  return (!(book->status & all_flags) && (book->status & init_flag)) ? 1 : 0;
}

int pdicr_book_have_id (PDICR_Book *book) {
  return (book->version >= PDICR_Version_NEWDIC4) ? 1 : 0;
}

void pdicr_book_initialize (PDICR_Book *book) {
  int i;

  book->status = init_flag;
  book->cache = NULL;
  book->coding = PDICR_Coding_UNDECIDED;
  book->coding_name = NULL;
  book->file = NULL;
  book->path = NULL;
  /* book->pron_table = NULL; */
  book->pron_table_code = PDICR_Pron_Table_Code_ASIS;
  book->version = PDICR_Version_UNDECIDED;
  for (i = 0; i < 8; i++) book->id[i] = 0;
}

void pdicr_book_finalize (PDICR_Book *book) {
  if (pdicr_book_is_initialized(book)) {
    if (book->path) pdicr_free(book->path);
    if (book->file) fclose(book->file);
    if (book->coding_name) pdicr_free(book->coding_name);
    /* if (book->pron_table) pdicr_free((char *)book->pron_table); */
    if (book->cache) pdicr_caches_finalize(book->cache);
  }

  pdicr_book_initialize(book);
}

PDICR_Error_Code pdicr_book_bind (PDICR_Book *book, const char *path) {
  FILE *fp;
  PDICR_Error_Code result = PDICR_SUCCESS;

  if (pdicr_book_is_bound(book)) pdicr_book_finalize(book);

  if (strlen(path) > PDICR_MAX_PATH_LENGTH) {
    result = PDICR_ERR_TOO_LONG_PATH;
    goto finished;
  }

  result = pdicr_strdup(path, &book->path);
  if (result != PDICR_SUCCESS) return result;
  
  book->file = fopen (path, "rb");
  if (book->file == NULL) {
    result = PDICR_ERR_BAD_FILE_NAME;
    goto finished;
  }

  result = bind_internal (book);
  
 finished:
  if (result != PDICR_SUCCESS)
    pdicr_book_finalize(book);
  else
    book->status |= bound_flag;
  return result;
}

static PDICR_Error_Code bind_internal (PDICR_Book *book) {
  PDICR_Error_Code ret = PDICR_SUCCESS;
  char header[256];
  int tmp;

  ret = pdicr_caches_fread(book, 0, 256, header, NULL);
  if (ret != PDICR_SUCCESS) return ret;

  switch (header[0x8d]) {
  case 4:
    book->version = PDICR_Version_NEWDIC3;
    break;
  case 5:
    book->version = PDICR_Version_NEWDIC4;
    break;
  case 6:
    if (header[0x8c] >= 10)
      book->version = PDICR_Version_NEWDIC51;
    else 
      book->version = PDICR_Version_NEWDIC5;
    break;
  default: 
    return PDICR_ERR_UNSUPPORTED_BOOK;
  }

  /* Get Dictionary coding. */
  if (header[140 + 2 * 10 + 4 + 1] & 0x08) {
    book->coding = PDICR_Coding_BOCU1;
    book->pron_table_code = PDICR_Pron_Table_Code_ASIS;
  } else if (header[140 + 2 * 10 + 4 + 1] & 0x10)
    /* UTF16 dictionary */
    return PDICR_ERR_UNSUPPORTED_BOOK;
  else {
    book->pron_table_code = PDICR_Pron_Table_Code_SILIPA93;
    switch (header[171]) {
    case 0x00:
    case 0x01:
    case 0x02:
      book->coding = PDICR_Coding_SJIS;
      break;
    case 0x03:
      book->coding = PDICR_Coding_EUC;
      break;
    case 0x04:
      book->coding = PDICR_Coding_JIS;
      break;
    default:
      return PDICR_ERR_UNSUPPORTED_BOOK;
    }
  }

  /* Get book id */
  if (book->version >= PDICR_Version_NEWDIC4)
    memcpy(book->id, header + 0xd8, 8);

  /* Get dictionary defaults */
  ret = pdicr_book_default(book);

  /* Get max length of heading. */
  book->heading_max_length = pdicr_uint2(header + 142);
  
  /* Get position of index and data. */
  book->block_size = pdicr_uint2(header + 140 + 2 * 3);

  if (book->coding == PDICR_Coding_BOCU1)
    tmp = pdicr_uint4(header + 184);
  else
    tmp = pdicr_uint4(header + 182);

  book->index_start = pdicr_uint2(header + 140 + 2 * 5) + tmp;

  book->data_start = book->index_start
    + book->block_size * pdicr_uint2(header + 140 + 2 * 4);

  /* Get index order. */
  book->index_order = (book->version >= PDICR_Version_NEWDIC5) ?
    PDICR_Index_Order_Dictionary : header[140 + 2 * 10 + 4];

  /* Get size of block number (2 or 4) */
  if (book->coding == PDICR_Coding_BOCU1)
    book->block_number_size = header[182] * 2 + 2;
  else
    book->block_number_size = header[198] * 2 + 2;
  if (book->block_number_size != 2 && book->block_number_size != 4) {
    return PDICR_ERR_UNSUPPORTED_BOOK;
  }

  return PDICR_SUCCESS;
}

PDICR_Error_Code pdicr_book_path (PDICR_Book *book, char **path) {
  if (!pdicr_book_is_bound(book)) return PDICR_ERR_UNBOUND_BOOK;
  return pdicr_strdup(book->path, path);
}

PDICR_Error_Code pdicr_book_version
(PDICR_Book *book, PDICR_Version *version) {
  if (!pdicr_book_is_bound(book)) return PDICR_ERR_UNBOUND_BOOK;
  *version = book->version;
  return PDICR_SUCCESS;
}

PDICR_Error_Code pdicr_book_character_code
(PDICR_Book *book, PDICR_Character_Code *character_code) {
  if (!pdicr_book_is_bound(book)) return PDICR_ERR_UNBOUND_BOOK;

  *character_code =  book->coding;
  return PDICR_SUCCESS;
}

PDICR_Error_Code pdicr_book_id (PDICR_Book *book, char *id) {
  if (!pdicr_book_is_bound(book)) return PDICR_ERR_UNBOUND_BOOK;
  if (!pdicr_book_have_id(book)) return PDICR_ERR_NO_ID;

  memcpy(id, book->id, PDICR_BOOK_ID_LENGTH);
  return PDICR_SUCCESS;
}

PDICR_Error_Code pdicr_book_index_order
(PDICR_Book *book, PDICR_Index_Order *order) {
  if (!pdicr_book_is_bound(book)) return PDICR_ERR_UNBOUND_BOOK;

  *order = book->index_order;
  return PDICR_SUCCESS;
}

PDICR_Error_Code pdicr_book_pron_table_coding
(PDICR_Book *book, PDICR_Pron_Table_Code *pron_coding) {
  if (!pdicr_book_is_bound(book)) return PDICR_ERR_UNBOUND_BOOK;

  *pron_coding = book->pron_table_code;
  return PDICR_SUCCESS;
}

PDICR_Error_Code pdicr_book_set_pron_table_coding
(PDICR_Book *book, PDICR_Pron_Table_Code pron_coding) {
  if (!pdicr_book_is_bound(book)) return PDICR_ERR_UNBOUND_BOOK;

  switch (pron_coding) {
  case PDICR_Pron_Table_Code_ASIS:
  case PDICR_Pron_Table_Code_SILIPA93:
    book->pron_table_code = pron_coding;
    return PDICR_SUCCESS;
  default:
    break;
  }

  return PDICR_ERR_INVALID_PRON_TABLE_CODE;
}


PDICR_Error_Code pdicr_book_fread
(PDICR_Book *book, PDICR_Position pos, size_t length, char *buffer) {
  if (!pdicr_book_is_bound(book)) return PDICR_ERR_UNBOUND_BOOK;

  return pdicr_caches_fread(book, pos, length, buffer, NULL);
}
