;;; ndebs.el --- Another eblook interface
;; Copyright (C) 2006 Kazuhiro Ito <kzhr@d1.dion.ne.jp>

;; Author: Kazuhiro Ito <kzhr@d1.dion.ne.jp>

;; ndebs.el is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; ndebs.el is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with Lookup; if not, write to the Free Software Foundation,
;; Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;;; Commentary:
;; ndebs.el is copied and modified from ndeb.el included in Lookup.

;;; Code:

(require 'ndeb)

(defconst ndebs-version "0.1")

;;;
;;; Customizable variables
;;;

(defgroup ndebs nil
  "Lookup ndebs interface."
  :group 'lookup-agents)

(defcustom ndebs-program-arguments '("-q" "-e" "euc-jp")
  "*A list of arguments for eblook."
  :type '(repeat (string :tag "option"))
  :group 'ndebs)

(defcustom ndebs-process-coding-system
  (if (featurep 'evi-mule) (evi-coding-system 'euc-jp))
  "*Coding system for eblook process."
  :type 'symbol
  :group 'ndebs)

(defcustom ndebs-title-use-ascii t
  "*Non-nil, convert title to one that use ascii characters."
  :type 'boolean
  :group 'ndebs)


;;;
;;; Internal variables
;;;

(defvar ndebs-process nil
  "Process object for ndebs agents.")

(defvar ndebs-status nil
  "process stataus cache.")

(defvar ndebs-vars nil
  "process variables cache.")


;;;
;:: types
;;;

(put 'ndebs ':methods (get 'ndeb ':methods))
(put 'ndebs ':gaiji-regexp (get 'ndeb ':gaiji-regexp))
(put 'ndebs ':reference-pattern (get 'ndeb ':reference-pattern))
(put 'ndebs ':reference-regexp (get 'ndeb ':reference-regexp))
(put 'ndebs ':ignore-regexp  (get 'ndeb ':ignore-regexp))
(put 'ndebs ':headings (get 'ndeb ':headings))
(put 'ndebs ':arranges (get 'ndeb ':arranges))
(put 'ndebs ':adjusts (get 'ndeb ':adjusts))
(put 'ndebs ':xbm-regexp (get 'ndeb ':xbm-regexp))
(put 'ndebs ':bmp-regexp (get 'ndeb ':bmp-regexp))
(put 'ndebs ':jpeg-regexp (get 'ndeb ':jpeg-regexp))
(put 'ndebs ':wave-regexp (get 'ndeb ':wave-regexp))
(put 'ndebs ':mpeg-regexp (get 'ndeb ':mpeg-regexp))


;;;
;:: Interface functions
;;;

(put 'ndebs 'setup 'ndebs-setup)
(defun ndebs-setup (agent)
  (ndebs-start-process)
  (ndebs-select-book agent)
  (let ((string (ndebs-require "list")) (index 0) name title dicts)
    (while (string-match "^[^.]+\\. \\([^\t]+\\)\t\\(.*\\)" string index)
      (setq name (match-string 1 string)
	    title (match-string 2 string)
	    index (match-end 0))
      (when ndebs-title-use-ascii
	(setq title (japanese-hankaku title t)))
      (setq dicts (cons (lookup-new-dictionary  agent name name title) dicts)))
    (nreverse dicts)))

(put 'ndebs 'clear 'ndebs-clear)
(defun ndebs-clear (agent)
  (when ndebs-process
    (when (eq (process-status ndebs-process) 'run)
      (process-send-string ndebs-process "quit\n"))
    (lookup-process-kill ndebs-process)))

(put 'ndebs 'menu 'ndebs-dictionary-menu)
(defun ndebs-dictionary-menu (dictionary)
  (ndebs-select-dictionary dictionary)
  (list (lookup-make-entry dictionary "menu" "[Menu]")))

(put 'ndebs 'copyright 'ndebs-dictionary-copyright)
(defun ndebs-dictionary-copyright (dictionary)
  (ndebs-select-dictionary dictionary)
  (list (lookup-make-entry dictionary "copyright" "[Copyright]")))

(put 'ndebs 'search 'ndebs-dictionary-search)
(defun ndebs-dictionary-search (dictionary query)
  (ndebs-select-dictionary dictionary)
  (if lookup-max-hits (ndebs-require-set "max-hits" lookup-max-hits))
  (ndebs-require-set "decorate-mode" "off")
  (let ((method (lookup-query-method query)))
    (ndebs-require-set "search-method"
		       (lookup-assq-ref ndeb-method-table method))
    (ndebs-require
     (format "search \"%s\"" (lookup-query-string query))
     (lambda (process)
       (let (code heading entries dupchk)
	 (while (re-search-forward "^[^.]+\\. \\([^\t]+\\)\t\\(.*\\)" nil t)
	   (setq code (match-string 1) heading (match-string 2))
	   ;; $BF1$8%(%s%H%j$,$"$k$+%A%'%C%/$9$k!#(B
	   ;; $B$3$l$,$1$C$3$&$"$k$s$@!&!&(B
	   (unless (member (cons code heading) dupchk)
	     (setq entries
		   (cons (lookup-make-entry dictionary code heading) entries))
	     (setq dupchk (cons (cons code heading) dupchk))))
	 (nreverse entries))))))

(put 'ndebs 'content 'ndebs-dictionary-content)
(defun ndebs-dictionary-content (dictionary entry)
  (ndebs-select-dictionary dictionary)
  (if lookup-max-text (ndebs-require-set "max-text" lookup-max-text))
  (ndebs-require-set "stop-code"
		     (lookup-dictionary-option dictionary ':stop-code t))
  (let ((code (lookup-entry-code entry)) return)
    (ndebs-require-set "decorate-mode" "on")
    (setq return
	  (cond ((string= code "menu")
		 (ndebs-require "menu"))
		((string= code "copyright")
		 (ndebs-require "copyright"))
		(t
		 (ndebs-require (concat "content "
					(lookup-entry-code entry))))))
    return))

(put 'ndebs 'gaiji 'ndebs-dictionary-gaiji)
(defun ndebs-dictionary-gaiji (dictionary code)
  (ndebs-select-dictionary dictionary)
  (let ((xbm (ndebs-require (concat "font " code))))
    (if (string-match "default_width" xbm)
	xbm)))

;;;
;;; Internal functions
;;;

(defun ndebs-start-process ()
  "eblook$B$,5/F0$7$F$$$J$1$l$P5/F0$9$k!#(B"
  (unless (and (processp ndebs-process)
	       (eq (process-status ndebs-process) 'run))
    (let ((buffer (or (lookup-open-process-buffer " *ndebs*")
		      (lookup-temp-buffer))))
      (setq ndebs-process (apply 'start-process "ndebs" buffer
				 ndeb-program-name ndebs-program-arguments)
	    ndebs-status nil
	    ndebs-vars nil)
      (set-process-coding-system ndebs-process
				 ndebs-process-coding-system
				 ndebs-process-coding-system)
      (process-kill-without-query ndebs-process)
      (with-current-buffer buffer
	(catch 'started
	  (while (accept-process-output ndebs-process 10)
	    (save-excursion
	      (goto-char (point-min))
	      (when (search-forward ndeb-prompt-string nil t)
		(throw 'started t))))
	  (error "Failed start process")))
      (unless lookup-debug-mode
	(set-process-buffer ndebs-process nil)
	(kill-buffer buffer)))))

(defun ndebs-select-book (agent)
  "eblook$B$N(Bbook$B$r(Bagent$B$KJQ99$9$k!#4{$KA*Br$5$l$F$$$k>l9g$O2?$b$7$J$$!#(B"
  (let ((book (lookup-agent-location agent))
	(appendix (lookup-agent-option agent ':appendix)))
    (unless (and (equal (ndebs-status-get 'book) book)
		 (equal (ndebs-status-get 'appendix) appendix))
      (let ((ret-string (ndebs-require (concat "book " book " " appendix))))
	(unless (equal ret-string "")
	  (error ret-string)))
      (ndebs-status-set 'book book)
      (ndebs-status-set 'appendix appendix)
      (ndebs-status-set 'subbook nil))))

(defun ndebs-select-dictionary (dictionary)
  "$B<-=q$r(Bdictionary$B$KJQ99$9$k!#4{$KA*Br$5$l$F$$$k>l9g$O2?$b$7$J$$!#(B"
  (ndebs-select-book (lookup-dictionary-agent dictionary))
  (let ((subbook (lookup-dictionary-code dictionary)))
    (unless (equal (ndebs-status-get 'subbook) subbook)
      (ndebs-require (concat "select " subbook))
      (ndebs-status-set 'subbook subbook))))

(defun ndebs-status-get (key)
  (lookup-assq-ref ndebs-status key))

(defun ndebs-status-set (key value)
  (setq ndebs-status
	(lookup-assq-set ndebs-status key value)))

(defun ndebs-vars-get (var)
  (lookup-assoc-ref ndebs-vars var))

(defun ndebs-vars-set (var value)
  (setq ndebs-vars
	(lookup-assoc-set ndebs-vars var value)))

(defun ndebs-require (string &optional filter)
  "eblook$B$K%3%^%s%I$rAw?.$7!"=PNO$5$l$?J8;zNs$rJV$9!#(Blookup-process-require$B$r;2>H!#(B"
  (lookup-process-require ndebs-process (concat string "\n")
			  (concat "^" ndeb-prompt-string) filter))
  
(defun ndebs-require-set (var value)
  "eblook$B$NJQ?t$r(Bset$B$^$?$O(B(unset)$B$9$k!#4{$K(Bset$B$5$l$F$$$k>l9g$O2?$b$7$J$$!#(B"
  (unless (and (equal (ndebs-vars-get var) value) value)
    (ndebs-vars-set var value)
    (if value
	(ndebs-require (format "set %s \"%s\"" var value))
      (ndebs-require (format "unset %s" var)))))

(provide 'ndebs)

;;; ndsrd.el ends here
