;;; ndeb.el --- Lookup eblook interface
;; Copyright (C) 1999 Lookup Development Team <lookup@ring.gr.jp>

;; Author: Keisuke Nishida <kei@psn.net>
;; Version: $Id: ndeb.el,v 1.2.4.31 2008/12/08 09:23:46 kazuhiro Exp $

;; This file is part of Lookup.

;; Lookup is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; Lookup is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with Lookup; if not, write to the Free Software Foundation,
;; Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;;; Code:

(require 'lookup)

(defconst ndeb-version "1.2")

;;;
;:: Customizable variables
;;;

(defgroup ndeb nil
  "Lookup eblook interface."
  :group 'lookup-agents)

(defcustom ndeb-program-name "eblook"
  "*Program name of eblook."
  :type 'string
  :group 'ndeb)

(defcustom ndeb-program-arguments '("-q" "-e" "euc-jp" "-i")
  "*A list of arguments for eblook."
  :type '(repeat (string :tag "option"))
  :group 'ndeb)

(defcustom ndeb-prompt-string "eblook> "
  "*Prompt string of eblook."
  :type 'string
  :group 'ndeb)

(defcustom ndeb-process-coding-system
  (if (featurep 'evi-mule) (evi-coding-system 'euc-jp))
  "*Coding system for eblook process."
  :type 'symbol
  :group 'ndeb)

(defcustom ndeb-max-image-size 1048576
  "*ndeb $B8!:w;~$KI=<($9$k2hA|$N:GBg%5%$%:(B ($B%P%$%H(B)$B!#(B
$B%G%U%)%k%H$O(B 1MB$B!#(Bnil $B$r;XDj$9$k$H!"L5@)8B$K$J$k!#(B"
  :type 'integer
  :group 'ndeb)

(defcustom ndeb-gaiji-size 16
  "$B%G%U%)%k%H$G;HMQ$9$k30;z$N%5%$%:!#;XDj$7$?%5%$%:$N30;z$,B8:_$7$J$$>l9g$O;XDjCM$r1[$($J$$:GBg%5%$%:$r!"$=$l$bB8:_$7$J$$>l9g$O(B16$B%I%C%H$N30;z$r;HMQ$9$k!#(B"
  :type '(choice :tag "size"
		 (const 16)
		 (const 24)
		 (const 30)
		 (const 48))
  :group 'ndeb)

(defcustom ndeb-minimum-indent 1
  "$B;XDj$7$??t;z$r1[$($?J,$@$1;z2<$2=hM}$r9T$&!#DL>o$O(B0$B$^$?$O(B1$B!#(B"
  :type 'integer
  :group 'ndeb)


;;;
;:: Internal varialbes
;;;

(defvar ndeb-current-agent nil)
(defvar ndeb-current-dictionary nil)
(defvar ndeb-current-process nil)

(defconst ndeb-method-table
  '((exact . "exact") (prefix . "word") (suffix . "endword") (wild . "wild")))

(defconst ndeb-gaiji-default-size 16)

;;;
;:: types
;;;

;; ndeb agent:
;;
;;   (ndeb DIRECTORY :appendix APPENDIX)
;;
;; DIRECTORY - dictionary directory
;; APPENDIX  - appendix directory
;;
;; [property]
;; ndeb-process - eblook process related with agent
;; ndeb-dict    - last used dictionary
;; ndeb-method  - last used value of search-method
;; ndeb-stop    - last used value of stop-code

(defun ndeb-agent-directory (agent)
  (let ((dir (lookup-agent-location agent)))
    (if dir 
	(if (string-match "^ebnet://" dir)
	    dir
	  (expand-file-name dir))
      (error "You should specify a dictionary directory"))))

(defun ndeb-agent-appendix (agent)
  (let ((dir (lookup-agent-option agent ':appendix)))
    (if dir
	(if (string-match "^ebnet://" dir)
	    dir
	  (expand-file-name dir)))))

(defun ndeb-agent-coding (agent)
  (or (lookup-agent-option agent ':coding)
      ndeb-process-coding-system))

(require 'ndeb-binary)

(put 'ndeb ':methods '(exact prefix suffix keyword substring wild))
(put 'ndeb ':gaiji-regexp "<gaiji=\\([^>]*\\)>")
(put 'ndeb ':reference-pattern '("<reference>\\($B"*(B?\\(\\(.\\|\n\\)*\\)\\)</reference=\\([^>]+\\)>" 1 2 4))
(put 'ndeb ':reference-regexp '("<reference>" . "</reference=[^>]+>"))
(put 'ndeb ':ignore-regexp "\\(</?su[pb]>\\|</?em>\\|<ind=[0-9]>\\)")

(put 'ndeb ':headings '(lookup-arrange-gaijis))

(put 'ndeb ':arranges
     '(ndeb-arrange-auto-jump-reference
       ndeb-arrange-xbm
       ndeb-arrange-bmp
       ndeb-arrange-jpeg
       ndeb-arrange-image-page
       ndeb-arrange-wave
       ndeb-arrange-mpeg
       ndeb-arrange-indent
       ndeb-arrange-scripts
       ndeb-arrange-ignore
       ndeb-arrange-no-newline
       ndeb-arrange-prev-next
       ndeb-arrange-paged-reference
       lookup-arrange-references
       lookup-arrange-gaijis
       lookup-arrange-squeezed-references
       lookup-arrange-default-headings
       ndeb-arrange-fill-lines
       ndeb-arrange-snd-autoplay))

(put 'ndeb ':adjusts
     '(lookup-adjust-check-references
       lookup-adjust-show-gaijis
       lookup-adjust-goto-min))

;; ndeb dictionary:
;;
;; CODE - same as NAME below
;; NAME - given by eblook `list' command
;; 
;; [option]
;; :coding    - process coding system
;; :stop-code - stop-code used by eblook

(defun ndeb-new-dictionary (name title)
  (lookup-new-dictionary ndeb-current-agent name name title))

(defun ndeb-dictionary-coding (dictionary)
  (or (lookup-dictionary-option dictionary ':coding t)
      ndeb-process-coding-system))

(defun ndeb-dictionary-stopcode (dictionary)
  (lookup-dictionary-option dictionary ':stop-code t))

;; ndeb entry:
;;
;; CODE    - entry specific code (e.g. "2c00:340") by eblook `search' command
;; HEADING - given by eblook `search' command

(defun ndeb-make-entry (code heading)
  (lookup-make-entry ndeb-current-dictionary code heading))

;;;
;:: macros
;;;

(put 'ndeb-with-agent 'lisp-indent-function 1)
(defmacro ndeb-with-agent (agent &rest body)
  (` (let ((ndeb-current-agent (, agent))
	   (ndeb-current-process (ndeb-agent-process (, agent))))
       (,@ body))))

(put 'ndeb-with-dictionary 'lisp-indent-function 1)
(defmacro ndeb-with-dictionary (dictionary &rest body)
  (` (ndeb-with-agent (lookup-dictionary-agent (, dictionary))
       (let ((ndeb-current-dictionary (, dictionary)))
	 (unless (eq (, dictionary)
		     (lookup-agent-get-property ndeb-current-agent 'ndeb-dict))
	   ;; $BI,MW$J$H$-$@$1<-=q$r(B select $B$9$k!#(B
	   ;; $B30It%W%m%;%9$H$d$j$H$j$9$k$h$j$3$NJ}$,9bB.$@$m$&$7!"(B
	   ;; $B%G%P%C%0$N$H$-%P%C%U%!$,$4$A$c$4$A$c$9$k$N$O$&$6$C$?$$!#(B
	   (ndeb-process-send
	    (concat "select " (lookup-dictionary-code (, dictionary))))
	   (lookup-agent-put-property ndeb-current-agent 'ndeb-dict
				      (, dictionary))
	   ;; $B<-=qKh$KJ8;z%3!<%I$r@_Dj$9$k!#(B
	   (let ((code (ndeb-dictionary-coding (, dictionary))))
	     (when code
	       (set-process-coding-system ndeb-current-process code code))))
	 (,@ body)))))

(defun ndeb-agent-process (agent)
  (let ((process (lookup-agent-get-property agent 'ndeb-process)))
    (unless (and process (eq (process-status process) 'run))
      (if process (lookup-process-kill process))
      (setq process (ndeb-process-open (ndeb-agent-directory agent)
				       (ndeb-agent-appendix agent)))
      ;; $B:G=i$K<-=q0lMw$rF@$k$N$KJ8;z%3!<%I$N@_Dj$,I,MW!#(B
      (let ((coding (ndeb-agent-coding agent)))
	(when coding
	  (set-process-coding-system process coding coding)))
      ;; $B%3%^%s%I$N<B9TKh$K9T$J$&I,MW$N$"$k=hM}!#(B
      (let ((ndeb-current-process process))
	(if lookup-max-hits (ndeb-require-set "max-hits" lookup-max-hits))
	(if lookup-max-text (ndeb-require-set "max-text" lookup-max-text)))
      (lookup-agent-put-property agent 'ndeb-process process)
      (lookup-agent-put-property agent 'ndeb-dict nil)
      (lookup-agent-put-property agent 'ndeb-method nil)
      (lookup-agent-put-property agent 'ndeb-stop nil))
    process))

(defun ndeb-agent-kill-process (agent)
  (let ((process (lookup-agent-get-property agent 'ndeb-process)))
    (when process
      (if (eq (process-status process) 'run)
	  (process-send-string process "quit\n"))
      (lookup-process-kill process)
      (lookup-agent-put-property agent 'ndeb-process nil))))


;;;
;:: Interface functions
;;;

(put 'ndeb 'setup 'ndeb-setup)
(defun ndeb-setup (agent)
  (ndeb-with-agent agent
    (ndeb-process-require "list"
      (lambda (process)
	(let (name title dicts)
	  (while (re-search-forward "^[^.]+\\. \\([^\t]+\\)\t\\(.*\\)" nil t)
	    (setq name (match-string 1) title (match-string 2))
	    (setq dicts (cons (ndeb-new-dictionary name title) dicts)))
	  (nreverse dicts))))))

(put 'ndeb 'clear 'ndeb-clear)
(defun ndeb-clear (agent)
  (ndeb-agent-kill-process agent)
  (ndeb-binary-clear agent))

(put 'ndeb 'menu 'ndeb-dictionary-menu)
(defun ndeb-dictionary-menu (dictionary)
  (ndeb-with-dictionary dictionary
    (let ((rtn (ndeb-process-require "subinfo"))
	  entries)
      (setq rtn (and (string-match "^ search methods:\\( .+\\)$" rtn)
		     (match-string 1 rtn)))
      (when rtn
	(when (string-match  " menu\\($\\| \\)" rtn)
	  (setq entries (list (ndeb-make-entry "menu" "[Menu]"))))
	(when (string-match  " image_menu\\($\\| \\)" rtn)
	  (setq entries
		(append entries
			(list
			 (ndeb-make-entry "image_menu" "[Graphic menu]"))))))
      entries)))


(put 'ndeb 'copyright 'ndeb-dictionary-copyright)
(defun ndeb-dictionary-copyright (dictionary)
  (ndeb-with-dictionary dictionary
    (list (ndeb-make-entry "copyright" "[Copyright]"))))

(put 'ndeb 'search 'ndeb-dictionary-search)
(defun ndeb-dictionary-search (dictionary query)
  (ndeb-with-dictionary dictionary
    (let ((method (lookup-query-method query))
	  (string (lookup-query-string query))
	  (last (lookup-agent-get-property ndeb-current-agent 'ndeb-method))
	  cmd)
      (cond
       ((eq method 'keyword)
	(let (qstring)
	  (setq qstring string)
	  (while (string-match "[ \t$B!!(B]+" qstring)
	    (setq qstring (replace-match "=" nil t qstring)))
	  (setq cmd 
		(format "set search-method keyword\nsearch \"=%s\"\n"
			(ndeb-escape-query qstring)))
	  
	  (setq qstring string)
	  (while (string-match "[ \t$B!!(B]+" qstring)
	    (setq qstring (replace-match "&" nil t qstring)))
	  (setq cmd 
		(concat cmd (format "set search-method cross\nsearch \"&%s\""
				    (ndeb-escape-query qstring))))
	  (lookup-agent-put-property ndeb-current-agent 'ndeb-method "cross")))
       (t
	(when (eq method 'substring)
	  (setq method 'wild
		string (concat "*" string "*")))
	(unless (eq method last)
	  ;; $BI,MW$N$"$k$H$-$@$1(B search-method $B$r@_Dj$9$k!#(Bndeb-dict $B$KF1$8!#(B
	  (ndeb-require-set "search-method"
			    (lookup-assq-ref ndeb-method-table method))
	  (lookup-agent-put-property ndeb-current-agent 'ndeb-method method))
	(setq cmd (format "search \"%s\"" (ndeb-escape-query string)))))
      (ndeb-process-require cmd
        (lambda (process)
	  (let (code heading dupchk entries)
	    (while (re-search-forward "^[^.]+\\. \\([^\t]+\\)\t\\(.*\\)" nil t)
	      (setq code (match-string 1) heading (match-string 2))
	      ;; $BF1$8%(%s%H%j$,$"$k$+%A%'%C%/$9$k!#(B
	      ;; $B$3$l$,$1$C$3$&$"$k$s$@!&!&(B
	      (unless (member (cons code heading) dupchk)
		(setq entries (cons (ndeb-make-entry code heading) entries))
		(setq dupchk (cons (cons code heading) dupchk))))
	    (nreverse entries)))))))

(put 'ndeb 'content 'ndeb-dictionary-content)
(defun ndeb-dictionary-content (dictionary entry)
  (ndeb-with-dictionary dictionary
    (let ((stop (ndeb-dictionary-stopcode dictionary))
          (last (lookup-agent-get-property ndeb-current-agent 'ndeb-stop)))
      (unless (eq stop last)
        ;; $BI,MW$N$"$k$H$-$@$1(B stop-code $B$r@_Dj$9$k!#(Bndeb-dict $B$KF1$8!#(B
        (ndeb-require-set "stop-code" stop)
        (lookup-agent-put-property ndeb-current-agent 'ndeb-stop stop)))
    (let ((code (lookup-entry-code entry)) return)
      (ndeb-require-set "decorate-mode" "on")
      (setq return
	    (if (member code '("menu" "image_menu" "copyright"))
		(ndeb-process-require code)
	      (ndeb-process-require (concat "content "
					    (lookup-entry-code entry)))))
      (ndeb-require-set "decorate-mode" "off")
      return)))
	  

(put 'ndeb 'gaiji 'ndeb-dictionary-gaiji)
(defun ndeb-dictionary-gaiji (dictionary code)
  (ndeb-with-dictionary dictionary
    (when (null lookup-use-bitmap)
      (let (height tmp)
	(setq height (lookup-dictionary-option dictionary ':gaiji-size t))
	(unless height
	  (with-temp-buffer
	    (insert-string (ndeb-process-require "subinfo"))
	    (when (search-backward "font sizes:")
	      (while (re-search-forward "[0-9]+" nil t)
		(setq tmp (string-to-number (match-string 0)))
		(if (eq tmp ndeb-gaiji-size)
		    (progn
		      (setq height tmp)
		      (goto-char (point-max)))
		  (when (and (> ndeb-gaiji-size tmp)
			     (or (null height)
				 (< height tmp)))
		    (setq height tmp))))))
	  (setq height (or height ndeb-gaiji-default-size))
	  (lookup-set-dictionary-option (lookup-dictionary-id dictionary)
					:gaiji-size height))
	(ndeb-require-set "font" (number-to-string height))))
    (let ((xbm (ndeb-process-require (concat "font " code))))
      (catch ':done
	(when (string-match "default_width" xbm)
	  (throw ':done xbm))
	(when (or lookup-use-bitmap
		  (equal (ndeb-process-require "show font") "16"))
	  (throw ':done nil))
	(ndeb-require-set "font" "16")
	(setq xbm (ndeb-process-require (concat "font " code)))
	(when  (string-match "default_width" xbm)
	  xbm)))))


;;;
;:: Internal functions
;;;

(defun ndeb-require-set (var value)
  (if value
      (ndeb-process-send (format "set %s \"%s\"" var value))
    (ndeb-process-send (format "unset %s" var))))

(defun ndeb-escape-query (string)
  (let ((start 0))
    (while (string-match "\\\\" string start)
      (setq string (replace-match "\\\\" t t string)
	    start (1+ (match-end 0)))))
  string)

(defun ndeb-arrange-no-newline (entry)
  (while (search-forward "<no-newline>" nil t)
    (let ((beg-beg (match-beginning 0))
	  (beg-end (match-end 0)))
      (if (and (re-search-forward "<\\(/?\\)no-newline>" nil t)
	       (equal (match-string 1) "/"))
	  (let ((end-beg (match-beginning 0)))
	    (goto-char end-beg)
	    (skip-chars-backward "\t " beg-end)
	    (when (> (point) (point-min))
	      (backward-char))
	    (when (< beg-end (point))
	      (add-text-properties beg-end (point) '(read-only t)))
	    (delete-region end-beg (match-end 0))
	    (delete-region beg-beg beg-end))
	(goto-char beg-beg)
	(delete-region beg-beg beg-end)))))

(defun ndeb-arrange-prev-next (entry)
  (while (re-search-forward "\\(<prev>\\|<next>\\)" nil t)
    (if (equal (match-string 0) "<prev>")
	(replace-match "\n($BA09`L\"M(B")
      (replace-match "($B<!9`L\"M(B"))
    (if (re-search-forward "\\(</prev>\\|</next>\\)" nil t)
	(replace-match ")"))))

(defun ndeb-arrange-ignore (entry)
  (let ((regexp (lookup-dictionary-option dictionary ':ignore-regexp t)))
    (while (re-search-forward regexp nil t)
      (replace-match ""))))

(defalias 'ndeb-arrange-fill-lines 'lookup-arrange-fill-lines)

(defun ndeb-arrange-paged-reference (entry)
  (while (re-search-forward "<paged-reference=\\([0-9]+:[0-9]+\\)>" nil t)
    (let ((pos (match-string 1))
	  (start (match-beginning 0))
	  (end (match-end 0)))
      (condition-case nil
	  (progn
	    (search-forward "</paged-reference>")
	    (replace-match (format "</reference=%s>" pos))
	    (delete-region start end)
	    (goto-char start)
	    (insert "<reference>"))
	(error nil)))))

(defun ndeb-arrange-auto-jump-reference (entry)
  (when (re-search-forward "<auto-jump-reference></auto-jump-reference=\\([0-9]+:[0-9]+\\)>" nil t)
    (let ((code (match-string 1)))
      (delete-region (point-min) (point-max))
      (insert
       (if (eq (lookup-agent-class (lookup-dictionary-agent dictionary)) 'ndeb)
	   (ndeb-dictionary-content dictionary (ndeb-make-entry code nil))
	 (ndebs-dictionary-content
	  dictionary (lookup-make-entry dictionary code nil)))))))

(defun ndeb-arrange-indent (entry)
  (while (re-search-forward "<ind=\\([0-9]\\)>" nil t)
    (let ((beg-beg (match-beginning 0))
	  (beg-end (match-end 0))
	  (level (- (string-to-number (match-string 1))
		    (or 
		     (lookup-dictionary-option dictionary ':minimum-indent t)
		     ndeb-minimum-indent)))
	  indent-end point)
      (delete-region beg-beg (point))
      (when (> level 0)
	(setq point (point))
	(setq indent-end
	      (or (and (re-search-forward "<ind=[0-9]>" nil t)
		       (match-beginning 0))
		  (point-max)))
	(set-left-margin point indent-end level)
	(goto-char point)))))

(defun ndeb-arrange-scripts (entry)
  (while (re-search-forward "<\\(su[bp]\\)>" nil t)
    (let ((beg-beg (match-beginning 0))
	  (tag (match-string 1)))
      (delete-region beg-beg (match-end 0))
      (if (and (re-search-forward (concat "<\\(/?\\)" tag ">") nil t)
	       (equal (match-string 1) "/"))
	  (let ((end-beg (match-beginning 0)))
	    (goto-char end-beg)
	    (add-text-properties beg-beg (point)
				 `(display ((raise ,(if (equal tag "sub")
							-0.3
						      0.3))
					    (height (- 2)))))
	    (delete-region end-beg (match-end 0)))
	(goto-char beg-beg)))))


;;;
;:: eblook process
;;;

(defun ndeb-process-open (directory appendix)
  (let* ((args (append ndeb-program-arguments
		       (cons directory (if appendix (list appendix)))))
	 (buffer (lookup-open-process-buffer (concat " *ndeb+" directory "*")))
	 (process (apply 'start-process "ndeb"
			 (or buffer (lookup-temp-buffer))
			 ndeb-program-name args)))
    (process-kill-without-query process)
    (accept-process-output process)
    (with-current-buffer (process-buffer process)
      (save-excursion
	(goto-char (point-min))
	(if (search-forward "Warning: invalid book directory" nil t)
	    (error "Invalid dictionary directory: %s" directory))
	(goto-char (point-min))
	(if (search-forward "Warning: invalid appendix directory" nil t)
	    (error "Invalid appendix directory: %s" appendix)))
      (process-send-string process "set prompt\n")
      (insert "set prompt\n")
      (unless buffer
	(set-process-buffer process nil)
	(kill-buffer (current-buffer))))
    process))

(put 'ndeb-process-require 'lisp-indent-function 1)
(defun ndeb-process-require (command &optional filter)
  (let ((lookup-process-output-separator-lines 0))
    (lookup-process-require ndeb-current-process
			    (concat command "\nset prompt \""
				    ndeb-prompt-string "\"\nset prompt\n")
			    (concat "^" ndeb-prompt-string) filter)))

(defun ndeb-process-send (string)
  (lookup-process-send ndeb-current-process (concat string "\n")))

(provide 'ndeb)

;;; ndeb.el ends here
