#include <../config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdint.h>

#include <zlib.h>

#include "defs.h"

/* global varaibles */
extern int global_verbose;
extern int global_debug;
extern int global_keep_original;
extern int global_overwrite;
extern int global_stdout;

/* common.c */
uint32_t read_uint32(unsigned char *buffer);
uint16_t read_uint16(unsigned char *buffer);
void debug_out (const char *message, ...);
void verbose_out (const char *message, ...);

/* internal functions*/
static enum dzzip_error_code decompress_file_internal
(int infile, int outfile, off_t start, size_t length, char *header);
static enum dzzip_error_code is_dictzip_header(int infile, char *header);
static enum dzzip_error_code extract_file_internal
(int infile, int outfile, off_t start, size_t length, struct stat *in_stat);
static enum dzzip_error_code skip_comment(int infile, int count, off_t *start);

enum dzzip_error_code decompress_file (const char *in_filename, off_t start, size_t length) {
  int infile = -1, outfile = -1;
  off_t end;
  char *out_filename = NULL;
  struct stat in_stat, out_stat;
  enum dzzip_error_code error_code = DZZIP_SUCCESS;
  char header[DICTZIP_HEADER_SIZE];

  if (global_stdout) {
    outfile = 1;
  } else {
    for (int i = strlen(in_filename) - 1; i > 0; i--) {
      if (in_filename[i] == '.') {
	if (out_filename = malloc(i + 1)) {
	  memcpy(out_filename, in_filename, i);
	  out_filename[i] = 0;
	} else {
	  error_code = DZZIP_ERR_MEMORY_EXHAUSTED;
	  goto finished;
	}
	break;
      }
    }
    if (!out_filename) {
      error_code = DZZIP_ERR_INVALID_INPUT_FILE_NAME;
      goto finished;
    }
    verbose_out("Output filename is %s", out_filename);
    if (stat(out_filename, &out_stat) == 0) {
      if (S_ISREG(out_stat.st_mode) && global_overwrite) {
	if (unlink(out_filename) == -1) {
	  error_code = DZZIP_ERR_FILE_IO;
	  goto finished;
	}
      } else {
	error_code = DZZIP_ERR_OUTPUT_EXISTS;
	goto finished;
      }
    }

    debug_out("Opening output file %s", out_filename);
    outfile = open(out_filename, O_RDWR | O_CREAT | O_BINARY, 0666);
    if (outfile == -1) {
      error_code = DZZIP_ERR_FILE_IO;
      goto finished;
    }
  }

  infile = open(in_filename, O_RDONLY | O_BINARY);
  if (infile == -1) {
    error_code = DZZIP_ERR_FILE_IO;
    goto finished;
  }

  error_code = is_dictzip_header(infile, header);
  switch (error_code) {
  case DZZIP_SUCCESS:
    error_code = decompress_file_internal
      (infile, outfile, start, length, header);
    break;
  case DZZIP_ERR_NOT_DICTZIP:
    if (stat(in_filename, &in_stat)) {
      error_code = DZZIP_ERR_FILE_IO;
      goto finished;
    }
    error_code = extract_file_internal
      (infile, outfile, start, length, &in_stat);
  default:
    goto finished;
  }

 finished:
  if (infile != -1) close(infile);
  if (!(outfile == -1 || global_stdout)) close(outfile);
  if (out_filename) free(out_filename);
  return error_code;
}

static enum dzzip_error_code is_dictzip_header (int infile, char *header) {
  ssize_t length;
  enum dzzip_error_code error_code = DZZIP_SUCCESS;
  
  debug_out("Check dictzip header");

  if (lseek(infile, 0, SEEK_SET) == -1) {
    error_code = DZZIP_ERR_FILE_IO;
    goto finished;
  }

  length = read(infile, header, DICTZIP_HEADER_SIZE);
  if (length == -1) {
    error_code = DZZIP_ERR_FILE_IO;
  } else if (length < DICTZIP_HEADER_SIZE) {
    error_code = DZZIP_ERR_NOT_DICTZIP;
  } else if(!(header[0] == 31 /* ID1 */ &&
	      header[1] == (char)139 /* ID2 */ &&
	      header[2] == 8 /* CM */ &&
	      (header[3] & (char)0xe4) == 4 /* FLG */ &&
	      (header[8] == 2 || header [8] ==4) /* XFL */ &&
	      header[12] == 'R' /* SI1 */ &&
	      header[13] == 'A' /* SI2 */ &&
	      read_uint16(header + 16) == 1 /* VER */)) {
    error_code = DZZIP_ERR_NOT_DICTZIP;
  }

 finished:
  switch (error_code) {
  case DZZIP_SUCCESS:
    debug_out("Input file is dictzipped file.");
    break;
  case DZZIP_ERR_NOT_DICTZIP:
    debug_out("Input file is NOT dictzipped file.");
    break;
  }
  return error_code;
}

#define DZZIP_HEADER_STRING_READ_BUFFER_SIZE 1024

static enum dzzip_error_code decompress_file_internal
(int infile, int outfile, off_t start, size_t length, char *header) {
  enum dzzip_error_code error_code = DZZIP_SUCCESS;
  char tail[4];
  size_t chunk_size;
  int chunk_count, start_chunk, end_chunk;
  char *header_buffer = NULL, *in_buffer = NULL, *out_buffer = NULL;
  off_t *chunk_offsets = NULL;
  uint16_t *chunk_sizes = NULL;
  int zlib_init = 0;
  off_t end;
  size_t decompressed_size;

  if (lseek(infile, -4, SEEK_END) == -1 ||
      read(infile, tail, 4) != 4 ||
      lseek(infile, 0, SEEK_SET) == -1) {
    error_code = DZZIP_ERR_FILE_IO;
    goto finished;
  }
  decompressed_size = read_uint32(tail);
  debug_out("original file size is %d.", decompressed_size);

  if (start >= decompressed_size) {
    error_code = DZZIP_ERR_INVALID_START;
    goto finished;
  }

  end = length ? (start + length -1) : decompressed_size - 1;
  if (end > decompressed_size - 1) end = decompressed_size - 1;
  debug_out("start position is %d, end position is %d.", start, end);

  chunk_size = read_uint16(header + 18);
  chunk_count = read_uint16(header + 20);
  debug_out("chunk size is 0x%x, chunk_count is 0x%x",
	    chunk_size, chunk_count);
  
  start_chunk = start / chunk_size;
  end_chunk = (end + 1) / chunk_size;
  debug_out("start chunk is 0x%x, end chunk is 0x%x", start_chunk, end_chunk);

  chunk_sizes = malloc((end_chunk + 1) * 2);
  chunk_offsets = malloc((end_chunk + 1) * sizeof(chunk_offsets[0]));
  if (!chunk_offsets || !chunk_sizes) {
    error_code = DZZIP_ERR_MEMORY_EXHAUSTED;
    goto finished;
  }

  lseek(infile, DICTZIP_HEADER_SIZE, SEEK_SET);
  if (read(infile, chunk_sizes, (end_chunk + 1) * 2) != (end_chunk + 1) * 2) {
    error_code = DZZIP_ERR_FILE_IO;
    goto finished;
  }

  chunk_offsets[0] = DICTZIP_HEADER_SIZE + chunk_count * 2;
  if (header[3] & 0x18) {
    error_code = skip_comment
      (infile, ((header[3] & 0x18) == 0x18) ? 2 : 1, &(chunk_offsets[0]));
    if (error_code != DZZIP_SUCCESS) {
      goto finished;
    }
  }

  chunk_offsets[0] += ((header[3] & 0x02) ? 2 : 0);
  for (int i = 0; i < end_chunk; i++) {
    chunk_sizes[i] = read_uint16(((unsigned char *)chunk_sizes) + i * 2);
    chunk_offsets[i + 1] = chunk_offsets[i] + chunk_sizes[i];
  }

  z_stream stream;
  stream.zalloc = Z_NULL;
  stream.zfree = Z_NULL;
  stream.opaque = Z_NULL;
  if (inflateInit2(&stream, -15) != Z_OK)
    goto zlib_failed;
  zlib_init = 1;
  
  in_buffer=malloc(DICTZIP_MAX_CHUNK_SIZE);
  out_buffer=malloc(chunk_size);
  if (!in_buffer || !out_buffer) {
    error_code = DZZIP_ERR_MEMORY_EXHAUSTED;
    goto finished;
  }

  for (int i = start_chunk; i <= end_chunk; i++) {
    debug_out("chunk %x/%x, offset %x, compressed size is %x",
	      i, end_chunk, chunk_offsets[i], chunk_sizes[i]);
    if ((lseek(infile, chunk_offsets[i], SEEK_SET) == -1) ||
	(read(infile, in_buffer, chunk_sizes[i]) != chunk_sizes[i])) {
      error_code = DZZIP_ERR_FILE_IO;
      goto finished;
    }
    stream.next_in = in_buffer;
    stream.avail_in = chunk_sizes[i];
    stream.next_out = out_buffer;
    stream.avail_out = chunk_size;
    if (inflate(&stream, Z_PARTIAL_FLUSH) != Z_OK) goto zlib_failed;
    size_t decompressed_size = chunk_size - stream.avail_out;

    int write_start = (i == start_chunk) ? (start % chunk_size) : 0;
    size_t write_size =
      ((i == end_chunk) ? ((end % chunk_size) + 1) : decompressed_size)
      - write_start;
    if (write(outfile, out_buffer + write_start, write_size)
	!= (ssize_t)write_size) {
      error_code = DZZIP_ERR_FILE_IO;
      goto finished;
    }
  }
  goto finished;

 zlib_failed:
  fprintf(stderr, "zlib error, %s\n", stream.msg);
  error_code = DZZIP_ERR_ZLIB_ERROR;
 finished:
  if (zlib_init) inflateEnd(&stream);
  if (!in_buffer) free(in_buffer);
  if (!out_buffer) free(out_buffer);
  if (!chunk_offsets) free(chunk_offsets);
  if (!chunk_sizes) free(chunk_sizes);
  if (!header_buffer) free(header_buffer);
  return error_code;
}

#define DZZIP_EXTRACT_BUFFER_SIZE (2048)

static enum dzzip_error_code extract_file_internal
(int infile, int outfile, off_t start, size_t length, struct stat *in_stat) {
  char *buffer;
  off_t end;
  enum dzzip_error_code error_code;
  int start_chunk, end_chunk;

  if (start >= in_stat->st_size) {
    error_code = DZZIP_ERR_INVALID_START;
    goto finished;
  }

  end = length ? (start + length -1) : in_stat->st_size - 1;
  if (end > in_stat->st_size - 1) end = in_stat->st_size - 1;
  debug_out("start position is %d, end position is %d.", start, end);

  start_chunk = start / DZZIP_EXTRACT_BUFFER_SIZE;
  end_chunk = (end + 1) / DZZIP_EXTRACT_BUFFER_SIZE;

  buffer=malloc(DZZIP_EXTRACT_BUFFER_SIZE);
  if (!buffer) {
    error_code = DZZIP_ERR_MEMORY_EXHAUSTED;
    goto finished;
  }

  for (int i = start_chunk; i <= end_chunk; i++) {
    size_t decompressed_size;
    if ((lseek(infile, i * DZZIP_EXTRACT_BUFFER_SIZE, SEEK_SET) == -1) ||
	(read(infile, buffer, DZZIP_EXTRACT_BUFFER_SIZE) !=
	 DZZIP_EXTRACT_BUFFER_SIZE)) {
      error_code = DZZIP_ERR_FILE_IO;
      goto finished;
    }

    int write_start = (i == start_chunk) ?
      (start % DZZIP_EXTRACT_BUFFER_SIZE) : 0;
    size_t write_size =
      ((i == end_chunk) ?
       ((end % DZZIP_EXTRACT_BUFFER_SIZE) + 1) : DZZIP_EXTRACT_BUFFER_SIZE) -
      write_start;
    if (write(outfile, buffer + write_start, write_size)
	!= (ssize_t)write_size) {
      error_code = DZZIP_ERR_FILE_IO;
      
      goto finished;
    }
  }

 finished:
  if (!buffer) free(buffer);
  return error_code;
}


static enum dzzip_error_code skip_comment
(int infile, int count, off_t *start) {
  debug_out("skipping comment and filename...");
  off_t shift = 0;
  int i;
  ssize_t read_bytes;
  char *header_buffer = malloc(DZZIP_HEADER_STRING_READ_BUFFER_SIZE);
  enum dzzip_error_code error_code = DZZIP_SUCCESS;

  if (!header_buffer) {
    error_code = DZZIP_ERR_MEMORY_EXHAUSTED;
    goto finished;
  }

  if (lseek(infile, *start + shift, SEEK_SET) == -1) {
    error_code = DZZIP_ERR_FILE_IO;
    goto finished;
  }

  while(count) {
    read_bytes = read
      (infile, header_buffer, DZZIP_HEADER_STRING_READ_BUFFER_SIZE);
    switch (read_bytes) {
    case -1:
      error_code = DZZIP_ERR_FILE_IO;
      goto finished;
    case 0:
      error_code = DZZIP_ERR_INVALID_DICTZIP;
      goto finished;
    default:
      break;
    }

    for (i=0; i < read_bytes && count; i++) {
      if(header_buffer[i] == 0) {
	count--;
      }
    }
    shift += count ? read_bytes : i;
  }

  debug_out("header comment and filename length is %d bytes.", shift);
  *start += shift;

 finished:
  if (header_buffer) free(header_buffer);
  return error_code;
}
