#include <../config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdint.h>

#include <zlib.h>

#include "defs.h"

/* global varaibles */
extern int global_verbose;
extern int global_debug;
extern int global_keep_original;
extern int global_overwrite;
extern size_t chunk_size;

/* common.c */
void write_uint32(unsigned char *buffer, uint32_t n);
void write_uint16(unsigned char *buffer, uint16_t n);
void debug_out (const char *message, ...);
void verbose_out (const char *message, ...);

/* internal functions */

static enum dzzip_error_code compress_file_internal
(int infile, struct stat *in_stat, int outfile, size_t chunk_size);
static void write_dictzip_header
(unsigned char *buffer, time_t mtime, size_t chunk_size, int chunk_count);
static enum dzzip_error_code check_file_size
(struct stat *in_stat, size_t chunk_size);
static char* get_out_filename (const char *in_filename);

enum dzzip_error_code compress_file
(const char *in_filename, size_t chunk_size) {
  int infile = -1, outfile = -1;
  char *out_filename = NULL;
  struct stat in_stat, out_stat;
  enum dzzip_error_code error_code = DZZIP_SUCCESS;

  debug_out("Chunk size is %x", chunk_size);

  if (stat(in_filename, &in_stat)) {
    error_code = DZZIP_ERR_INVALID_INPUT_FILE;
    goto failed;
  }

  if ((error_code = check_file_size(&in_stat, chunk_size)) != DZZIP_SUCCESS) {
    goto failed;
  }

  out_filename = get_out_filename(in_filename);
  if (!out_filename) {
    error_code = DZZIP_ERR_MEMORY_EXHAUSTED;
    goto failed;
  }

  if (stat(out_filename, &out_stat) == 0) {
    if (S_ISREG(out_stat.st_mode) && global_overwrite) {
      if (unlink(out_filename) == -1) {
	error_code = DZZIP_ERR_FILE_IO;
	goto failed;
      }
    } else {
      error_code = DZZIP_ERR_OUTPUT_EXISTS;
      goto failed;
    }
  }

  debug_out("Opening input file %s", in_filename);
  infile = open(in_filename, O_RDONLY | O_BINARY);
  if (infile == -1) {
    error_code = DZZIP_ERR_FILE_IO;
    goto failed;
  }

  debug_out("Opening output file %s", out_filename);
  outfile = open(out_filename, O_RDWR | O_CREAT | O_BINARY, 0666);
  if (outfile == -1) {
    error_code = DZZIP_ERR_FILE_IO;
    goto failed;
  }

  error_code = compress_file_internal(infile, &in_stat, outfile, chunk_size);

 failed:
  if (out_filename) free(out_filename);
  if (infile != -1) close(infile);
  if (outfile != -1) close(outfile);
  return error_code;
}

static char* get_out_filename (const char *in_filename) {
  static const char *dictzip_suffix = ".dz";
  static const size_t dictzip_suffix_length = 3;
  size_t length = strlen(in_filename);
  char * out_filename;
  
  out_filename = malloc (length + dictzip_suffix_length + 1);
  if (out_filename) {
    memcpy(out_filename, in_filename, length);
    memcpy(out_filename + length, dictzip_suffix, dictzip_suffix_length + 1);
    verbose_out("Output file name is %s", out_filename);
  }

  return out_filename;
}

static enum dzzip_error_code check_file_size
(struct stat *in_stat, size_t chunk_size) {
  enum dzzip_error_code error_code = DZZIP_SUCCESS;

  if (in_stat->st_size > chunk_size * DICTZIP_MAX_CHUNK_COUNT) {
    debug_out("Input file is too large.");
    size_t required_chunk_size =
      (in_stat->st_size + DICTZIP_MAX_CHUNK_COUNT - 1) /
      DICTZIP_MAX_CHUNK_COUNT;
    if (required_chunk_size <= DICTZIP_MAX_CHUNK_SIZE) {
      fprintf(stderr, "Retry with larger chunk size (%d bytes or above).\n",
	      required_chunk_size);
    }
    error_code = DZZIP_ERR_TOO_LARGE_INPUT_FILE;
  }

  return error_code;
}

static enum dzzip_error_code compress_file_internal
(int infile, struct stat *in_stat, int outfile, size_t chunk_size) {
  char *header, *in_buffer, *out_buffer;
  int chunk_count = (in_stat->st_size + chunk_size - 1) / chunk_size;
  size_t header_size = DICTZIP_HEADER_SIZE + chunk_count * 2;
  size_t read_count = 0;
  uLong crc32 = 0;
  z_stream stream;
  enum dzzip_error_code error_code = DZZIP_SUCCESS;
  int zlib_init = 0;
  
  if (lseek(infile, 0, SEEK_SET) == -1) {
    error_code = DZZIP_ERR_FILE_IO;
    goto failed;
  }

  stream.zalloc = Z_NULL;
  stream.zfree = Z_NULL;
  stream.opaque = Z_NULL;
  if (deflateInit2(&stream, Z_BEST_COMPRESSION, Z_DEFLATED, -15,
		   Z_BEST_COMPRESSION, Z_DEFAULT_STRATEGY) != Z_OK)
    goto zlib_failed;
  zlib_init = 1;

  header = malloc(header_size);
  in_buffer = malloc(chunk_size);
  out_buffer = malloc(DICTZIP_MAX_CHUNK_SIZE);
  if (!in_buffer || !out_buffer || !header) {
    error_code = DZZIP_ERR_MEMORY_EXHAUSTED;
    goto failed;
  }

  write_dictzip_header(header, in_stat->st_mtime, chunk_size, chunk_count);
  if (write(outfile, header, header_size) != header_size) {
    error_code = DZZIP_ERR_FILE_IO;
    goto failed;
  }

  for (int i = 0; i < chunk_count; i++) {
    size_t read_size = read(infile, in_buffer, chunk_size);
    if (read_size == -1) {
      error_code = DZZIP_ERR_FILE_IO;
      goto failed;
    }

    stream.next_in = (Bytef *) in_buffer;
    stream.avail_in = read_size;
    stream.next_out = (Bytef *) out_buffer;
    stream.avail_out = DICTZIP_MAX_CHUNK_SIZE;

    if (deflate(&stream, Z_FULL_FLUSH) != Z_OK)
      goto zlib_failed;
    size_t c_size = DICTZIP_MAX_CHUNK_SIZE - stream.avail_out;

    read_count += read_size;
    debug_out("chunk %5d is compressed into %d (%x) bytes.",
	      i, c_size, c_size);
    crc32 = crc32_z(crc32, in_buffer, read_size);
    write_uint16(header + DICTZIP_HEADER_SIZE + i * 2, c_size);
    if (write(outfile, out_buffer, c_size) != c_size) {
      error_code = DZZIP_ERR_FILE_IO;
      goto failed;
    }
  }

  stream.next_out = (Bytef *) out_buffer;
  stream.avail_out = DICTZIP_MAX_CHUNK_SIZE;

  if (deflate(&stream, Z_FINISH) != Z_STREAM_END) {
    goto zlib_failed;
  }
  size_t c_size = DICTZIP_MAX_CHUNK_SIZE - stream.avail_out;

  write_uint32(out_buffer + c_size,     crc32);
  write_uint32(out_buffer + c_size + 4, read_count);

  if (write(outfile, out_buffer, c_size + 8) != c_size + 8) {
    error_code = DZZIP_ERR_FILE_IO;
    goto failed;
  }

  if (lseek(outfile, DICTZIP_HEADER_SIZE, SEEK_SET) == -1) goto failed;
  if (write(outfile, header + DICTZIP_HEADER_SIZE, chunk_count * 2)
      != chunk_count * 2) {
    error_code = DZZIP_ERR_FILE_IO;
    goto failed;
  }

  goto failed;

 zlib_failed:
  fprintf(stderr, "zlib error, %s\n", stream.msg);
  error_code = DZZIP_ERR_ZLIB_ERROR;
 failed:
  if (zlib_init) deflateEnd(&stream);
  if (header) free(header);
  if (in_buffer) free(in_buffer);
  if (out_buffer) free(out_buffer);
  return error_code;
}

static void write_dictzip_header
(unsigned char *buffer, time_t mtime, size_t chunk_size, int chunk_count) {
  buffer[0] = 31; /* ID1 */
  buffer[1] = 139; /* ID2 */
  buffer[2] = 8; /* CM */
  buffer[3] = 5; /* FLG */
  if (mtime >= (time_t) 0 && mtime < (time_t) 0xffffffff) {
    write_uint32(buffer + 4, mtime); /* MTIME */
  } else {
    debug_out("write_dictzip_header: mtime is out of 32bit range.");
    *(uint32_t *)(buffer + 4) = 0;
  }
  buffer[8] = 2; /* XFL */
  buffer[9] = 3; /* OS */

  /* dictzip header */
  write_uint16(buffer + 10, chunk_count * 2 + 6 + 4); /* XLEN */
  buffer[12] = 'R'; /* SI1 */
  buffer[13] = 'A'; /* SI2 */
  write_uint16(buffer + 14, chunk_count * 2 + 6); /* LEN */
  write_uint16(buffer + 16, 1); /* VER */
  write_uint16(buffer + 18, chunk_size); /* CHLEN */
  write_uint16(buffer + 20, chunk_count); /* CHCNT */
}
