/*                                                            -*- C -*-
 * Copyright (c) 2010  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "pdicr_internal.h"

#include <string.h>

/*
  bit status
    0 is set; 
    1 is search_started;
    2 is search_ended;
   22 is initialized
*/

#define  init_flag (0x00400000)
#define   set_flag (0x00000001)
#define start_flag (0x00000002)
#define   end_flag (0x00000004)
#define all_flags (~(init_flag | set_flag | start_flag | end_flag))

/* static functions */
static PDICR_Error_Code pdicr_field_number_size
(PDICR_Book *book, int block_number, size_t *number);
static PDICR_Error_Code pdicr_search_prepare
(PDICR_Book *book, PDICR_Search *search);
static PDICR_Error_Code pdicr_dump_next
(PDICR_Book *book, PDICR_Search *search, PDICR_Entry *entry);
static PDICR_Error_Code pdicr_dump_prepare_internal
(PDICR_Book *book, PDICR_Search *search);


void pdicr_dump_initialize (PDICR_Search *search) {
  search->status = init_flag;
  search->last_string = NULL;
  search->index = 0;
  search->data = 0;
  search->field_number_size = 0;
  search->ibuffer = NULL;
  search->dbuffer = NULL;
  search->ibuffer_size = 0;
  search->dbuffer_size = 0;

  pdicr_dump_set_set(search);
}

void pdicr_dump_finalize (PDICR_Search *search) {
  if (pdicr_dump_is_set(search)) {
    if (search->last_string) pdicr_free(search->last_string);
    if (search->ibuffer)     pdicr_free(search->ibuffer);
    if (search->dbuffer)     pdicr_free(search->dbuffer);
  }
  pdicr_dump_initialize(search);
}

void pdicr_dump_set_set (PDICR_Search *search) {
  search->status |= set_flag;
}

int pdicr_dump_is_set (PDICR_Search *search) {
  return (pdicr_dump_is_initialized(search) && (search->status & set_flag))
    ? 1 : 0;
}

void pdicr_dump_set_started (PDICR_Search *search) {
  search->status |= start_flag;
}

int pdicr_dump_is_started (PDICR_Search *search) {
  return (pdicr_dump_is_set(search) && (search->status & start_flag))
    ? 1 : 0;
}

int pdicr_dump_is_initialized (PDICR_Search *search) {
  return (!(search->status & all_flags) && (search->status & init_flag))
    ? 1: 0;
}

void pdicr_dump_set_ended (PDICR_Search *search) {
  search->status |= end_flag;
  if (search->last_string) {
    pdicr_free(search->last_string);
    search->last_string = NULL;
  }
  if (search->ibuffer) {
    pdicr_free(search->ibuffer);
    search->ibuffer = NULL;
  }
  if (search->dbuffer) {
    pdicr_free(search->dbuffer);
    search->dbuffer = NULL;
  }
}

int pdicr_dump_is_ended (PDICR_Search *search) {
  return (pdicr_dump_is_started(search) && (search->status & end_flag))
    ? 1 : 0;
}

/* dump functions */
PDICR_Error_Code pdicr_dump_dump (PDICR_Book *book, PDICR_Search *search,
				  PDICR_Content *content) {
  pdicr_log("in: pdicr_dump_dump");

  PDICR_Position pos;
  PDICR_Error_Code ret = PDICR_SUCCESS;
  PDICR_Entry entry;

  if (!pdicr_book_is_bound(book)) {
    ret = PDICR_ERR_UNBOUND_BOOK;
    goto ret;
  }
  if (!pdicr_dump_is_set(search)) {
    ret = PDICR_ERR_INVALID_SEARCH;
    goto ret;
  }

  pdicr_entry_initialize(&entry);
  if (!pdicr_dump_is_ended(search) && 
      (ret = pdicr_dump_next (book, search, &entry)) != PDICR_SUCCESS)
    goto finish;
  
  if (pdicr_entry_is_set(&entry))
    ret = pdicr_content_load(book, &entry, content);
  else
    /* no hits */
    pdicr_content_finalize(content);

 finish:
  pdicr_entry_finalize(&entry);

 ret:
  pdicr_log("out: pdicr_dump_dump (return %d)", ret);
  return ret;
}

static PDICR_Error_Code pdicr_field_number_size
(PDICR_Book *book, int block_number, size_t *number) {
  pdicr_log("in: pdicr_field_number_size (block %d)", block_number);

  PDICR_Error_Code ret;
  char buffer;

  ret = pdicr_book_fread(book, book->data_start
			+ block_number * book->block_size + 1, 1, &buffer);
  if (ret != PDICR_SUCCESS) return ret;

  *number = (buffer & ((char) 0x80)) ? 4 : 2;

  pdicr_log("out: pdicr_field_number_size (return %d, field number size %d)",
	    ret, (int)*number);
  return ret;
}

static PDICR_Error_Code pdicr_dump_prepare
(PDICR_Book *book, PDICR_Search *search) {
  pdicr_log("in: pdicr_dump_prepare");

  PDICR_Error_Code ret = PDICR_SUCCESS;
  search->data = 0;
  search->index = book->index_start;

  ret = pdicr_malloc(book->heading_max_length + 1, &search->last_string);
  if (ret != PDICR_SUCCESS) goto failed;

  search->ibuffer_size = book->block_number_size
    + book->heading_max_length + 1;
  ret = pdicr_malloc(search->ibuffer_size + 1, &search->ibuffer);
  if (ret != PDICR_SUCCESS) goto failed;
  search->ibuffer[search->ibuffer_size] = 0;

  search->dbuffer_size = PDICR_MAX_FIELD_NUMBER_SIZE
    + book->heading_max_length + 2;
  ret = pdicr_malloc(search->dbuffer_size + 1, &search->dbuffer);
  if (ret != PDICR_SUCCESS) goto failed;
  search->dbuffer[search->dbuffer_size] = 0;
  ret = pdicr_dump_prepare_internal(book, search);
  goto finish;

 failed:
  if (search->last_string) {
    pdicr_free(search->last_string);
    search->last_string = NULL;
  }

 if (search->ibuffer) {
    pdicr_free(search->ibuffer);
    search->ibuffer = NULL;
  }

  if (search->dbuffer) {
    pdicr_free(search->dbuffer);
    search->dbuffer = NULL;
  }

 finish:
  pdicr_log("out: pdicr_dump_prepare (return %d)", ret);
  return ret;
}

static PDICR_Error_Code pdicr_dump_prepare_internal
(PDICR_Book *book, PDICR_Search *search) {
  pdicr_log("in: pdicr_dump_prepare_internal");

  char *ibuffer;
  int current_block, tmp;
  PDICR_Error_Code ret = PDICR_SUCCESS;

  ibuffer = search->ibuffer;
  ret = pdicr_book_fread(book, search->index, search->ibuffer_size, ibuffer);
  if (ret != PDICR_SUCCESS) goto finished;

  current_block = (book->block_number_size == 2) ?
    pdicr_uint2(ibuffer) : pdicr_uint4(ibuffer);

  if (ibuffer[book->block_number_size] == 0)
    /* last index */
    search->index = book->data_start;
  else
    search->index += book->block_number_size
      + strlen(ibuffer + book->block_number_size) + 1;
	
  ret = pdicr_field_number_size(book, current_block,
			       &search->field_number_size);
  if (ret != PDICR_SUCCESS) goto finished;

  search->data = book->data_start + current_block * book->block_size + 2;

  pdicr_dump_set_started(search);

 finished:
  pdicr_log("out: pdicr_dump_prepare_internal (return %d)", ret);
  return ret;
}

static PDICR_Error_Code pdicr_dump_next
(PDICR_Book *book, PDICR_Search *search, PDICR_Entry *entry) {
  pdicr_log("in: pdicr_dump_next");

  char *index_buffer, *data_buffer;
  int last_block, current_block, bocu_flag, tmp;
  PDICR_Error_Code ret = PDICR_SUCCESS;
  size_t field_length = 0;

  if (pdicr_dump_is_ended(search)) goto no_hits;

  if (!pdicr_dump_is_started(search)) {
    if ((ret = pdicr_dump_prepare(book, search)) != PDICR_SUCCESS)
      goto finished;
    if (pdicr_dump_is_ended(search)) goto no_hits;
  }
  index_buffer = search->ibuffer;
  data_buffer = search->dbuffer;

  bocu_flag = ((book->coding == PDICR_Coding_BOCU1) ? 1 : 0);

  while ((search->index < book->data_start) || search->data) {
    ret = pdicr_book_fread(book, search->data, search->dbuffer_size,
			  data_buffer);
    if (ret != PDICR_SUCCESS) goto finished;

    field_length = ((search->field_number_size == 2) ?
		    pdicr_uint2(data_buffer) : pdicr_uint4(data_buffer));

    if (!field_length) {
      /* to next block */ 
      if (search->index >= book->data_start) goto no_hits;

      pdicr_log("go to next block");

      ret = pdicr_book_fread
	(book, search->index, search->ibuffer_size, index_buffer);
      if (ret != PDICR_SUCCESS) goto finished;

      search->index += book->block_number_size
	+ strlen(index_buffer + book->block_number_size) + 1;

      current_block = (book->block_number_size == 2)
	? pdicr_uint2(index_buffer) : pdicr_uint4(index_buffer);
      if (index_buffer[book->block_number_size] == 0) goto no_hits;

      ret = pdicr_field_number_size(book, current_block,
				   &search->field_number_size);
      if (ret != PDICR_SUCCESS) goto finished;

      search->data = book->data_start + current_block * book->block_size + 2;
      continue;
    } else {
      field_length += bocu_flag;
    }

    strcpy(search->last_string +
	   (unsigned char)data_buffer[search->field_number_size],
	   data_buffer + search->field_number_size + 1 + bocu_flag);

    ret = pdicr_entry_set_raw(book, entry, search->last_string,
			      search->data, field_length,
			      search->field_number_size);
    /* prepare for next entry */
    search->data += search->field_number_size + 1 + field_length;
    goto finished;
  }

 no_hits:
  pdicr_dump_set_ended(search);
  pdicr_entry_finalize(entry);

 finished:
  pdicr_log("out: pdicr_dump_next (return %d)", ret);
  return ret;
}
