/*                                                            -*- C -*-
 * Copyright (c) 2010  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "pdicr_internal.h"
#include "bocu1.h"

#include <errno.h>
#include <string.h>

#ifdef HAVE_ICONV_H
#include <iconv.h>
#endif

/* static functions */
static PDICR_Error_Code pdicr_convert_internal
(const char *to_code, const char *from_code, const char *string,
 const size_t length, char **to_string, size_t *to_length);

PDICR_Error_Code pdicr_convert_to_utf8
(PDICR_Book *book, const char *string, size_t length, char **to_string) {
  pdicr_log("in: pdicr_convert_to_utf8");

  PDICR_Error_Code ret =  PDICR_SUCCESS;

  if (!length) {
    *to_string = NULL;
    goto finish;
  }

  switch(book->coding) {
  case PDICR_Coding_SJIS:
    ret = pdicr_convert_internal
      (PDICR_ICONV_NAME_UTF8, PDICR_ICONV_NAME_SJIS, string, length,
       to_string, NULL);
    goto finish;
  case PDICR_Coding_BOCU1:
    ret = pdicr_convert_bocu1_utf8 (string, to_string);
    goto finish;
  default:
    ret = PDICR_ERR_UNSUPPORTED_OPERATION;
    break;
  }

 finish:
  if (ret == PDICR_SUCCESS && *to_string)
    pdicr_log("out: pdicr_convert_to_utf8 (return %d, string %s)",
	      *to_string);
  else
    pdicr_log("out: pdicr_convert_to_utf8 (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_convert_from_utf8
(PDICR_Book *book, const char *string, char **to_string, size_t *length) {
  pdicr_log("in: pdicr_convert_from_utf8 (string %s)", string);

  PDICR_Error_Code ret = PDICR_SUCCESS;

  if (!string || !strlen(string)) {
    *to_string = NULL;
    if (length) *length = 0;
    goto finish;
  }

  switch(book->coding) {
  case PDICR_Coding_SJIS:
    ret = pdicr_convert_internal
      (PDICR_ICONV_NAME_SJIS, PDICR_ICONV_NAME_UTF8, string, strlen(string),
       to_string, length);
    goto finish;
  case PDICR_Coding_BOCU1:
    ret = pdicr_convert_utf8_bocu1(string, to_string);
    if (ret == PDICR_SUCCESS && length)
      *length = strlen(*to_string);
    goto finish;
  default:
    break;
  }

  ret = PDICR_ERR_UNSUPPORTED_OPERATION;

 finish:
  pdicr_log("out: pdicr_convert_from_utf8 (return %d)", ret);
  return ret;
}

static PDICR_Error_Code pdicr_convert_internal
(const char *to_code, const char *from_code, const char *string, const
 size_t length, char **to_string, size_t *to_length) {
  pdicr_log("in: pdicr_convert_internal");

  PDICR_Error_Code ret = PDICR_SUCCESS;
  iconv_t cd;
  int buffer_size, i = 0;
  char *dst, *buffer = NULL;
  size_t srcleft, dstleft, iconv_res;
  int res_length;
  const char *src;

  cd = iconv_open(to_code, from_code);
  if (cd == (iconv_t) -1) return PDICR_ERR_ICONV_OPEN_FAILURE;

  buffer_size = length + 1;
  ret = pdicr_malloc(buffer_size, &buffer);
  if (ret != PDICR_SUCCESS) goto finished;

  src = string;
  srcleft = length;
  dst = buffer;
  dstleft = buffer_size - 1;
  while (i < 2) {
    iconv_res = iconv(cd, (char **)&src, &srcleft, &dst, &dstleft);
    if (iconv_res == (size_t) -1) {
      if (errno == E2BIG) {
	dstleft += buffer_size;
	buffer_size += buffer_size;

	ret = pdicr_realloc(buffer_size, &buffer);
	if (ret != PDICR_SUCCESS) goto finished;
	dst = buffer + buffer_size - 1 - dstleft;
      } else {
	ret = PDICR_ERR_ICONV_FAILURE;
	goto finished;
      }
    } else {
      src = NULL;
      i++;
    }
  }

  *dst = 0;
  res_length = (int) (dst - buffer);
  ret = pdicr_malloc(res_length + 1, to_string);
  if (ret != PDICR_SUCCESS) goto finished;
  memcpy(*to_string, buffer, res_length + 1);
  if (to_length) *to_length = res_length;

 finished:
  if (buffer) pdicr_free(buffer);
  if (cd != (iconv_t) -1) iconv_close(cd);

  pdicr_log("out: pdicr_convert_internal (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_convert_bocu1_utf8 (const char *str, char **to_s) {
  pdicr_log("in: pdicr_convert_bocu1_utf8");

  PDICR_Error_Code ret = PDICR_SUCCESS;
  Bocu1Rx Rx;
  char *buffer = NULL;
  int res, consumed = 0;
  size_t buffer_length;

  Rx.prev = 0;
  Rx.count = 0;
  Rx.diff = 0;

  buffer_length = strlen(str) * 2;
  ret = pdicr_malloc(buffer_length, &buffer);
  if (ret != PDICR_SUCCESS) goto finished;

  while (*str != 0) {
    res = decodeBocu1(&Rx, (unsigned char)*str);
    if (res < -1) {
      ret = PDICR_ERR_BOCU1_DECODE_FAILURE;
      goto finished;
    }
    str++;
    if (res == -1)
      continue;

    while ((buffer_length - consumed) < utf8_length(res)) {
      buffer_length += buffer_length;
      ret = pdicr_realloc(buffer_length, &buffer);
      if (ret != PDICR_SUCCESS) goto finished;;
    }
    consumed += pdicr_write_utf8(res, buffer + consumed);
  }

  ret = pdicr_malloc(consumed + 1, to_s);
  if (ret != PDICR_SUCCESS) goto finished;

  memcpy(*to_s, buffer, consumed);
  (*to_s)[consumed] = 0;

 finished:
  if (buffer) pdicr_free(buffer);

  if (ret == PDICR_SUCCESS)
    pdicr_log("out: pdicr_convert_bocu1_utf8 (return %d, string %s)",
	      ret, to_s);
  else
    pdicr_log("out: pdicr_convert_bocu1_utf8 (return %d)", ret);
  return ret;
}

PDICR_Error_Code pdicr_convert_utf8_bocu1 (const char *str, char **to_s) {
  pdicr_log("in: pdicr_convert_utf8_bocu1 (string %s)", str);

  PDICR_Error_Code ret = PDICR_SUCCESS;
  int prev = 0, current, read = 0, res, buffer_size = 0, consumed = 0;
  size_t str_length = strlen(str);
  int req_size, i;
  char *buffer = NULL;

  buffer_size = str_length;
  ret = pdicr_malloc(buffer_size, &buffer);
  if (ret != PDICR_SUCCESS) goto finished;

  while (str[read] != 0) {
    read += pdicr_read_utf8(str + read, &current);
    if (current == 0) {
      ret = PDICR_ERR_BOCU1_ENCODE_FAILURE;
      goto finished;
    }
    res = encodeBocu1(&prev, current);
    if (!res) {
      ret = PDICR_ERR_BOCU1_ENCODE_FAILURE;
      goto finished;
    }

    req_size = (res >> 24) & 0x0f;
    if (req_size > 4) req_size = 4;
    while (buffer_size < req_size + consumed) {
      buffer_size += buffer_size;
      ret = pdicr_realloc(buffer_size, &buffer);
      if (ret != PDICR_SUCCESS) goto finished;
    }

    consumed += req_size;
    for(i=0; i < req_size; i++) {
      buffer[consumed - i - 1] = (unsigned char)(res & 0xff);
      res = res >> 8;
    }
  }

  ret = pdicr_malloc(consumed + 1, to_s);
  if (ret != PDICR_SUCCESS) goto finished;

  memcpy(*to_s, buffer, consumed);
  (*to_s)[consumed] = 0;

 finished:
  if (buffer) pdicr_free(buffer);

  pdicr_log("out: pdicr_convert_utf8_bocu1 (return %d)", ret);
  return ret;
}

int pdicr_write_utf8 (int code, char *buffer) {
  /* no error check */
  if (code < 0x80) {
    *buffer = (char) code;
    return 1;
  } else if (code < 0x800) {
    buffer[0] = (char)(( code >>  6)         | 0xc0);
    buffer[1] = (char)(( code        & 0x3f) | 0x80);
    return 2;
  } else if (code < 0x10000) {
    buffer[0] = (char)(( code >> 12)         | 0xe0);
    buffer[1] = (char)(((code >>  6) & 0x3f) | 0x80);
    buffer[2] = (char)(( code        & 0x3f) | 0x80);
    return 3;
  }
  {
    buffer[0] = (char)(( code >> 18)         | 0xf0);
    buffer[1] = (char)(((code >> 12) & 0x3f) | 0x80);
    buffer[2] = (char)(((code >>  6) & 0x3f) | 0x80);
    buffer[3] = (char)(( code        & 0x3f) | 0x80);
    return 4;
  }
}

int pdicr_read_utf8 (const char *buffer, int *code) {
  if (!(((unsigned char)*buffer) & 0x80)) {
    *code = *buffer;
    return 1;
  } else if (!(((unsigned char)*buffer) & 0x20)) {
    *code =
      ((((unsigned char)buffer[0]) & 0x1f) <<  6) +
      ((((unsigned char)buffer[1]) & 0x3f)      );
    return 2;
  } else if (!(((unsigned char)*buffer) & 0x10)) {
    *code =
      ((((unsigned char)buffer[0]) & 0x0f) << 12) +
      ((((unsigned char)buffer[1]) & 0x3f) <<  6) + 
      ((((unsigned char)buffer[2]) & 0x3f)      );
    return 3;
  } else if (!(((unsigned char)*buffer) & 0x08)) {
    *code =
      ((((unsigned char)buffer[0]) & 0x07) << 18) +
      ((((unsigned char)buffer[1]) & 0x3f) << 12) + 
      ((((unsigned char)buffer[2]) & 0x3f) <<  6) + 
      ((((unsigned char)buffer[3]) & 0x3f)      );
    return 4;
  }

  /* invalid string */
  *code = 0;
  return 0;
}
