/*                                                            -*- C -*-
 * Copyright (c) 2008, 2013  Kazuhiro Ito
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sufary.h>

#include "getopt.h"
#include "config.h"

extern int opterr;
extern int optind;
extern int optopt;
extern int optreset;
extern char *optarg;

#define MAX_HITS "100"
#define LINE_BUFFER_SIZE 2000

/* type defnition */
struct Command_Table {
  const char *command;
  int (* function) (int argc, char **argv);
  const char *param;
  const char *desc;
};

struct Variable_Struct {
  struct Variable_Struct *next;
  char *name;
  char *value;
};
typedef struct Variable_Struct Variable;

/* declaration of internal functions */
void output_help ();
void output_version ();

void initialize ();
void finalize ();
void read_input (int *argc, char ***argv);
void parse_input (char *input, int *argc, char ***argv);
int exec_command (int argc, char **argv);

int variable_set (const char *name, const char *value, int force);
int variable_unset (const char *name, int force);
char *variable_get (const char *name);
int variable_name_cmp (const char *s1, const char *s2);
void variable_show_all ();
void update_internal ();

int command_search_array  (int argc, char **argv);
int command_search_salook (int argc, char **argv);
int command_file     (int argc, char **argv);
int command_quit     (int argc, char **argv);
int command_show     (int argc, char **argv);
int command_rsearch  (int argc, char **argv);
int command_help     (int argc, char **argv);
int command_batch    (int argc, char **argv);
int command_set      (int argc, char **argv);
int command_unset    (int argc, char **argv);
int command_variable (int argc, char **argv);
int command_ignore   (int argc, char **argv);
int command_mode     (int argc, char **argv);

void concat_arguments (int argc, char **argv);

int sa_result_list_count (SA_RESULT_LIST *results);
void sa_string_fputs (SA_STRING *string, FILE *fp);

/* constants */
const char short_options[] = "hv";
const struct option long_options[] = {
  {"help",    no_argument,       NULL, 'h'},
  {"version", no_argument,       NULL, 'v'},
  {NULL, 0, NULL, 0}
};

const char program_name[] = "salook";
const char default_prompt[] = "ok\n";
const char quit_string[] = "bye.";

const struct Command_Table command_table_array[] ={
  {"search", command_search_array, "string", "search with string."},
  {"show",   command_show  , "",             "show results."},
  {"style",  command_ignore, "",     "ignored, for compatibility with array."},
  {"init",   command_ignore, "",     "ignored, for compatibility with array."},
  {"order",  command_ignore, "",     "ignored, for compatibility wirh array."},
  {NULL, NULL, NULL, NULL},
};

const struct Command_Table command_table_salook[] ={
  {"search" , command_search_salook, "string", "search with string."},
  {"rsearch", command_rsearch, "regexp", "regexp search with regexp."},
  {NULL, NULL, NULL, NULL},
};

const struct Command_Table command_table_common[] ={
  {"file",     command_file,     "text [array]", "open file."},
  {"mode",     command_mode,     "name",       "set mode to salook or array."},
  {"set",      command_set,      "name value",   "set internal variable."},
  {"unset",    command_unset,    "name",          "unset internal variable."},
  {"variable", command_variable, "[name]",       "show value of internal variable[s]."},
  {"batch",    command_batch,    "command",      "process command and return no prompt."},
  {"help",     command_help,     "",             "show help."},
  {"quit",     command_quit,     "",             "exit salook process."},
  {NULL, NULL, NULL, NULL},
};

/* internal variables */
SUFARY *array = NULL;
char linebuf[LINE_BUFFER_SIZE];
int in_batch = 0;
int max_hits = 0;
Variable *variables = NULL;
int default_max_hits;
SUF_RESULT result;
const struct Command_Table *command_table = command_table_array;

/* functions */
int main (int argc, char **argv) {
  int i, iargc;;
  char **iargv = NULL;

  for (;;) {
    i = getopt_long(argc, argv, short_options, long_options, NULL);
    if (i == -1)
      break;
    switch (i) {
    case 'h':
      output_help();
      exit(0);
    case 'v':
      output_version(program_name);
      exit(0);
    default:
      output_help();
      exit(-1);
    }
  }

  initialize();

  if (argc - optind > 0) {
    i = command_file (2, &(argv[optind - 1]));
  }

  for(i = 0; i != -1;) {
    iargv = NULL;
    read_input(&iargc, &iargv);
    i = exec_command(iargc, iargv);
    if (iargv) free(iargv);
  }

  finalize();
  return 0;
}


void initialize () {
  string_to_number(MAX_HITS, &default_max_hits);
  linebuf[LINE_BUFFER_SIZE - 1] = 0;

  result.stat = FAIL;

  variable_set("_version", VERSION, 1);
  variable_set("max_hits", MAX_HITS, 1);

  command_table = command_table_array;
  variable_set("_mode", "array", 1);
}

void finalize () {
  if (array) sa_close(array);
}

void output_help () {
  printf("Usage: %s [option...]\n", program_name);
  puts("Options:");
  puts("  -v, --version       show version information.");
  puts("  -h, --help          show this message.");
}

void output_version (char *program_name) {
  printf("%s version %s\n", program_name, VERSION);
  printf("Copyright (C) 2008 Kazuhiro Ito\n");
}

void read_input (int *argc, char ***argv) {
  int i;
  char *prompt;

  if (!in_batch) {
    prompt = variable_get("prompt");
    printf("%s", prompt ? prompt : default_prompt);
    fflush(stdout);
  } else 
    in_batch = 0;
  fgets(linebuf, LINE_BUFFER_SIZE, stdin);

  for (i = 0; i < (LINE_BUFFER_SIZE - 1); i++)
    if (linebuf[i] == 0x0d || linebuf[i] == 0x0a) {
      linebuf[i] = 0;
      break;
    }
  parse_input (linebuf, argc, argv);
}

void parse_input (char *input, int *argc, char ***argv) {
  int count, is_escaped, in_string;
  char *dst, *start;

  *argc = 0;
  count = is_escaped = in_string = 0;

  /* count argc */
  for(; input[count]; count++) {
    if (!is_escaped && input[count] == ' ') {
      in_string = 0;
    } else if (!is_escaped && input[count] == '\\') {
      is_escaped = 1;
    } else {
      is_escaped = 0;
      if (!in_string) {
	(*argc)++;
	in_string = 1;
      }
    }
  }

  if (*argc == 0) return;

  /* make argv */
  *argv = malloc(sizeof(char *) * (*argc));
  if (!*argv) {
    fputs("memory exhausted\n", stderr);
    *argc = 0;
    return;
  }

  *argc = 0;
  count = is_escaped = in_string = 0;
  dst = input;

  for (; input[count]; count++) {
    if (!is_escaped && input[count] == ' ') {
      if (in_string) {
	in_string = 0;
	(*argv)[(*argc) - 1] = start;
	*dst = '\0';
	dst++;
      }
    } else if (!is_escaped && input[count] == '\\') {
      is_escaped = 1;
    } else {
      if (!in_string) {
	(*argc)++;
	in_string = 1;
	start = dst;
      }

      if (is_escaped) {
	switch (input[count]) {
	case 'n':
	  input[count] = '\n';
	  break;
	default:
	  break;
	}
      }

      *dst = input[count];
      is_escaped = 0;
      dst++;
    }
  }

  if (in_string) {
    (*argv)[(*argc) - 1] = start;
    *dst = '\0';
  }

  /* for debug */
  /*
    for (count = 0; count < *argc; count++)
    printf("\"%s\",", (*argv)[count]);
    printf("%d arguments\n", *argc);
  */
  return;
}

int exec_command (int argc, char **argv) {
  int ret = 0, i;

  if (argc) {
    for (i = 0; command_table[i].command; i++) {
      if (!strcmp(command_table[i].command, argv[0])) {
	ret = command_table[i].function(argc, argv);
	break;
      }
    }

    if (!command_table[i].command) {
      for (i = 0; command_table_common[i].command; i++) {
	if (!strcmp(command_table_common[i].command, argv[0])) {
	  ret = command_table_common[i].function(argc, argv);
	  break;
	}
      }
      if (!command_table_common[i].command) {
	printf("unknown command, %s\n", argv[0]);
	ret = 1;
      }
    }
  }

  return ret;
}

int string_to_number (const char *string, int *number) {
  int i = 0;
  *number = 0;

  while (1) {
    if (string[i] < '0' || string[i] > '9') break;
    *number *= 10;
    *number += (string[i] - '0');
    i++;
  }

  return i;
}

int variable_set (const char *name, const char *value, int force) {
  Variable *current = variables;

  if (name[0] == '_' && !force) {
    printf("variable %s is not writable\n", name);
    return  1;
  }

  for (;current; current = current->next) {
    if (!variable_name_cmp(name, current->name)) {
      if (current->name[0] == '_' && !force) {
	printf("variable %s is not writable\n", current->name + 1);
	return  1;
      }
      if (current->value) free(current->value);
      current->value = strdup(value);
      if (!current->value) {
	fputs("memory exhausted\n", stderr);
	variable_unset(name, force);
	return 2;
      } else
	goto succeeded;
    }
  }

  current = malloc(sizeof(Variable));
  if (!current) goto failed;
  current->name = strdup(name);
  if (!current->name) goto failed;
  current->value = strdup(value);
  if (!current->value) goto failed;

  current->next = variables;
  variables = current;
 succeeded:
  update_internal();
  return 0;
 
 failed:
  fputs("memory exhausted\n", stderr);
  if (current) {
    if (current->name) free(current->name);
    if (current->value) free(current->value);
    free(current);
  }
  return 2;
}

int variable_unset (const char *name, int force) {
  Variable *current = variables, *prev = NULL;

  if (name[0] == '_' && !force) {
    printf("variable %s is not writable\n", name);
    return  1;
  }

  for (;current; current = current->next) {
    if (!variable_name_cmp(current->name, name)) {
      if (current->name[0] == '_' && !force) {
	printf("variable %s is not writable\n", current->name + 1);
	return  1;
      }

      if (prev) {
	prev->next = current->next;
      } else {
	variables = current->next;
      }
      if (current->name) free(current->name);
      if (current->value) free(current->value);
      free(current);
      update_internal();
      return 0;
    }
    prev = current;
  }

  return 1;
}

char *variable_get (const char *name) {
  Variable *current = variables;

  for (;current; current = current->next)
    if (!variable_name_cmp(current->name, name))
      return (current->value) ? current->value : NULL;

  return NULL;
}

int variable_name_cmp (const char *s1, const char *s2) {
  if (s1[0] == '_') s1++;
  if (s2[0] == '_') s2++;

  return strcmp(s1, s2);
}

void variable_show_all () {
  Variable *current = variables;
  int flag;

  while (current) {
    if (current->value) {
      flag = current->name[0] == '_' ? 1 : 0;
      printf("%s=%s\n", current->name + flag, current->value);
    }
    current = current->next;
  }
}

/* update internal condition according to variables. */
void update_internal() {
  char *string;

  /* update max_hits */
  string = variable_get("max_hits");
  if (string)
    string_to_number(string, &max_hits);
  if (!max_hits)
    max_hits = default_max_hits;
}

/*
  internal commands.
*/
int command_file (int argc, char **argv) {
  char *path = NULL;

  if (argc < 2) {
    puts("too few arguments");
    return 1;
  } else if (argc > 3) {
    puts("too many arguments");
    return 1;
  }

  result.stat = FAIL;

  if (array)
    sa_close(array);
  
  if (argc == 2)
    array = sa_open(argv[1], NULL);
  else /* argc == 3 */
    array = sa_open(argv[1], argv[2]);

  if (!array) {
    printf("failed to open files, %s", argv[1]);
    if (argc == 3) printf(", %s", argv[2]);
    puts(".");
    return 1;
  }

  return 0;
}

int command_quit (int argc, char **argv) {
  printf("%s\n", quit_string);
  return -1;
}

int command_search_array (int argc, char **argv) {
  int i, count;
  char *line;
  /* SUF_RESULT result; */

  if (argc < 2) {
    puts("too few arguments");
    return 1;
  }

  if (!array) {
    puts("array is not opened");
    return 1;
  }

  concat_arguments(argc, argv);
  result = sa_find(array, 0, (sa_get_array_size(array))  - 1, argv[1],
		   strlen(argv[1]), 0);

  if (result.stat != SUCCESS) {
    puts("no matching elements.");
    return 1;
  }

  count = result.right - result.left + 1;
  printf ("FOUND: %d\n", count);
  return 0;
}

int command_search_salook (int argc, char **argv) {
  int i, count;
  SA_STRING string;

  if (argc < 2) {
    puts("too few arguments");
    return 1;
  }

  if (!array) {
    puts("array is not opened");
    return 1;
  }

  concat_arguments(argc, argv);
  result = sa_find(array, 0, (sa_get_array_size(array))  - 1, argv[1],
		   strlen(argv[1]), 0);

  if (result.stat != SUCCESS) {
    puts("no matching elements.");
    return 1;
  }

  count = result.right - result.left + 1;
  printf ("FOUND: %d\n", count);

  count = (max_hits && max_hits < count) ?
    result.left + max_hits - 1 : result.right;
  for (i = result.left; i <= count; i++) {
    string = sa_seek_context_lines(array, sa_aryidx2txtptr(array, i), 0, 0);
    sa_string_fputs(&string, stdout);
  }
}

int command_rsearch (int argc, char **argv) {
  int i, count;
  SA_STRING string;
  SA_RESULT_LIST *results, *result;

  if (argc < 2) {
    puts("too few arguments");
    return 1;
  }

  if (!array) {
    puts("array is not opened.");
    return 1;
  }

  concat_arguments(argc, argv);
  results = sa_regex(array, 0, sa_get_array_size(array)  - 1, argv[1],
		     strlen(argv[1]));

  if (!results) {
    puts("no matching elements.");
    return 1;
  }

  count = sa_result_list_count(results);
  printf ("FOUND: %d\n", count);
  count = max_hits < count ? max_hits : count;

  for (i = 0, result = results; i < count; i++, result = result->next) {
    string = sa_seek_context_lines
      (array, sa_aryidx2txtptr(array, result->value), 0, 0);
    sa_string_fputs (&string, stdout);
  }

  sa_free_result_list(results);
  return 0;
}

int command_show (int argc, char **argv) {
  int i, count;
  SA_STRING string;

  if (result.stat != SUCCESS) {
    puts("last search does not succeed.");
    return 1;
  }

  count = result.right - result.left + 1;
  count = (max_hits && max_hits < count) ?
    result.left + max_hits - 1 : result.right;
  for (i = result.left; i <= count; i++) {
    string = sa_seek_context_lines(array, sa_aryidx2txtptr(array, i), 0, 0);
    sa_string_fputs (&string, stdout);
  }

  return 0;
}

int command_help (int argc, char **argv) {
  int i;

  printf("%s version %s\n\n", program_name, VERSION);

  for (i = 0; command_table[i].command; i++)
    printf("%-9s %-12s %s\n", command_table[i].command,
	   command_table[i].param, command_table[i].desc);

  for (i = 0; command_table_common[i].command; i++)
    printf("%-9s %-12s %s\n", command_table_common[i].command,
	   command_table_common[i].param, command_table_common[i].desc);

  puts("\nTo input ' ' and '\\' as a part of argument, escaping with '\\' is needed.");

  return 0;
}

int command_batch (int argc, char **argv) {
  if (argc < 2) {
    puts("too few arguments");
    return 1;
  }

  in_batch = 1;
  return exec_command (argc - 1, &(argv[1]));
}

int command_set (int argc, char **argv) {
  if (argc < 3) {
    puts("too few arguments");
    return 1;
  } else if (argc > 3) {
    puts("too many arguments");
    return 1;
  }

  return variable_set(argv[1], argv[2], 0);
}

int command_unset (int argc, char **argv) {
  if (argc < 2) {
    puts("too few arguments");
    return 1;
  } else if (argc > 2) {
    puts("too many arguments");
    return 1;
  }

  return variable_unset(argv[1], 0);
}

int command_variable (int argc, char **argv) {
  char *value;

  if (argc < 1) {
    puts("too few arguments");
    return 1;
  } else if (argc > 2) {
    puts("too many arguments");
    return 1;
  }

  if (argc == 2) {
    value = variable_get(argv[1]);
    if (value)
      printf("%s=%s\n", argv[1], value);
    else
      printf("%s is no set\n", argv[1]);

    return 0;
  } else {
    variable_show_all();
  }
  return 0;
}

int command_ignore (int argc, char **argv) {
  return 0;
}

int command_mode (int argc, char **argv) {
  if (argc < 2) {
    puts("too few arguments");
    return 1;
  } else if (argc > 2) {
    puts("too many arguments");
    return 1;
  }

  if (!strcmp(argv[1], "salook")) {
    command_table = command_table_salook;
    variable_set("_mode", "salook", 1);
  } else if (!strcmp(argv[1], "array")) {
    command_table = command_table_array;
    variable_set("_mode", "array", 1);
  } else {
    printf("unknown mode, %s\n", argv[1]);
    return 1;
  }

  return 0;
}

int sa_result_list_count(SA_RESULT_LIST *results) {
  int i;
  for (i = 0; results; i++)
    results = results->next;

  return i;
}

void sa_string_fputs (SA_STRING *string, FILE *fp) {
  fwrite(string->ptr, 1, string->len, fp);
  putc('\n', fp);
}

void concat_arguments (int argc, char **argv) {
  int i;

  for (i = 2; i < argc; i++) {
    *(argv[i] - 1) = ' ';
  }
}
